package com.paypal.hybris.backoffice.widgets.order.authorize;

import com.hybris.cockpitng.engine.WidgetInstanceManager;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zul.Label;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.util.List;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.*;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.anyInt;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalAuthorizationControllerTest {

    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String NUMBER_OF_THREE_AUTHORIZATION_MESSAGE = "You have 3 more authorization attempts left.";
    private static final String NUMBER_OF_ONE_AUTHORIZATION_MESSAGE = "You have 1 more authorization attempt left.";
    private static final String TWO_HUNDRED = "200";
    private static final String HUNDRED = "100";
    private static final String ORDER_CODE = "orderCode";
    private static final String BLANK = "";
    private static final String ZERO = "0";
    private static final String NAN = "NaN";
    private static final String DISPLAY_NAME = "displayName";

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private WidgetInstanceManager widgetInstanceManager;

    @Mock
    private OrderModel order;

    @Mock
    private CustomerModel customerModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;

    @Mock
    private PaymentTransactionEntryModel entryModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private Textbox orderCode;

    @Mock
    private Textbox customer;

    @Mock
    private Textbox amount;

    @Mock
    private Label saveOrderMessage;

    @Mock
    private ModelService modelService;

    @Mock
    private PayPalManualAuthorizationService payPalManualAuthorizationService;

    @Spy
    @InjectMocks
    private PayPalAuthorizationController unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        doNothing().when(unit).showMessageBox(anyString(), anyString(), anyInt(), anyString());
    }

    @Test
    public void shouldShowInfoMessage() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(TWO_HUNDRED);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);

        when(payPalManualAuthorizationService.doAuthorization(order, new BigDecimal(TWO_HUNDRED))).thenReturn(entryModel);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_SUCCESS)).thenReturn(MANUAL_AUTHORIZATION_SUCCESS);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_TITLE)).thenReturn(MANUAL_AUTHORIZATION_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldShowInfoMessageWhenSaveOrderTrueButAvailableAmountNotLessThanZero() {
        unit.setOrder(order);
        when(order.getTotalPrice()).thenReturn(500D);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(TWO_HUNDRED);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        when(payPalManualAuthorizationService.doAuthorization(order, new BigDecimal(TWO_HUNDRED))).thenReturn(entryModel);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel, transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel));

        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(transactionModel.getPlannedAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_SUCCESS)).thenReturn(MANUAL_AUTHORIZATION_SUCCESS);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_TITLE)).thenReturn(MANUAL_AUTHORIZATION_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldShowErrorMessage() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(amount.getValue()).thenReturn(TWO_HUNDRED);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);

        when(payPalManualAuthorizationService.doAuthorization(order, new BigDecimal(TWO_HUNDRED))).thenReturn(entryModel);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_ERROR)).thenReturn(MANUAL_AUTHORIZATION_ERROR);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_TITLE)).thenReturn(MANUAL_AUTHORIZATION_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldShowExceptionErrorMessageWhenAdapterExceptionThrown() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(TWO_HUNDRED);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);

        when(payPalManualAuthorizationService.doAuthorization(order, new BigDecimal(TWO_HUNDRED))).thenReturn(entryModel);
        when(entryModel.getTransactionStatus()).thenThrow(AdapterException.class);

        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_TITLE)).thenReturn(MANUAL_AUTHORIZATION_TITLE);

        unit.confirm();

        verify(order).setPaymentStatusType(PaymentStatusType.FAILED);
        verify(modelService).save(order);
        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenValueBlank() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(BLANK);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_EMPTY_AMOUNT)).thenReturn(MANUAL_AUTHORIZATION_EMPTY_AMOUNT);

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenValueZero() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(ZERO);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_ZERO_AMOUNT)).thenReturn(MANUAL_AUTHORIZATION_ZERO_AMOUNT);

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenValueNotNumber() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(NAN);
        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT)).thenReturn(MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT);

        unit.confirm();
    }


    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenSaveOrderFlowActiveAndAvailableAmountLessThanZero() {
        unit.setOrder(order);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(amount.getValue()).thenReturn(TWO_HUNDRED);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel, transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel));
        when(order.getTotalPrice()).thenReturn(300D);

        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(transactionModel.getPlannedAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(widgetInstanceManager.getLabel(MANUAL_AUTHORIZATION_ORDER_AMOUNT_EXCEEDED)).thenReturn(MANUAL_AUTHORIZATION_ORDER_AMOUNT_EXCEEDED);

        unit.confirm();
    }

    @Test
    public void initCreateAuthorizeRequestFormWhenSaveOrderFlowNotActive() {
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getUser()).thenReturn(customerModel);
        when(order.getTotalPrice()).thenReturn(300D);

        when(customerModel.getDisplayName()).thenReturn(DISPLAY_NAME);

        doNothing().when(widgetInstanceManager).setTitle(any());

        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);

        unit.initCreateAuthorizeRequestForm(order);

        verify(orderCode).setValue(ORDER_CODE);
        verify(customer).setValue(DISPLAY_NAME);
        verify(amount).setValue("300.0");
    }

    @Test
    public void initCreateAuthorizeRequestFormWhenSaveOrderFlowActive() {
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getUser()).thenReturn(customerModel);
        when(order.getTotalPrice()).thenReturn(300D);

        when(customerModel.getDisplayName()).thenReturn(DISPLAY_NAME);

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel, transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel));

        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(transactionModel.getPlannedAmount()).thenReturn(new BigDecimal(HUNDRED));

        doNothing().when(widgetInstanceManager).setTitle(any());

        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        unit.initCreateAuthorizeRequestForm(order);

        verify(orderCode).setValue(ORDER_CODE);
        verify(customer).setValue(DISPLAY_NAME);
        verify(amount).setValue("100.00");
        verify(saveOrderMessage).setValue(NUMBER_OF_THREE_AUTHORIZATION_MESSAGE);
    }

    @Test
    public void initCreateAuthorizeRequestFormWhenSaveOrderFlowActiveAndNumbersOfAvailableAuthorizationLessThanTwo() {
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getUser()).thenReturn(customerModel);
        when(order.getTotalPrice()).thenReturn(300D);

        when(customerModel.getDisplayName()).thenReturn(DISPLAY_NAME);

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel, transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel));

        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(transactionModel.getPlannedAmount()).thenReturn(new BigDecimal(HUNDRED));

        doNothing().when(widgetInstanceManager).setTitle(any());

        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        unit.initCreateAuthorizeRequestForm(order);

        verify(orderCode).setValue(ORDER_CODE);
        verify(customer).setValue(DISPLAY_NAME);
        verify(amount).setValue("100.00");
        verify(saveOrderMessage).setValue(NUMBER_OF_ONE_AUTHORIZATION_MESSAGE);
    }
}