package com.paypal.hybris.backoffice.util;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.enums.ConsignmentStatus;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import org.apache.commons.lang.StringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

@UnitTest
public class PayPalBackofficeUtilTest {
    private AutoCloseable closeable;

    private static final String FIFTY = "50";

    private static final String THIRTY = "30";

    private static final String SEVENTY = "70";

    private static final String HUNDRED = "100";

    @Mock
    private PaymentTransactionEntryModel capturedEntry;

    @Mock
    private PaymentTransactionModel transaction;

    @Mock
    private PaymentTransactionModel transactionTwo;

    @Mock
    private PaymentTransactionEntryModel authorizedEntry;

    @Mock
    private ConsignmentModel consignmentModel;

    @Before
    public void setUp() throws Exception {
        closeable = MockitoAnnotations.openMocks(this);
    }

    @Test
    public void shouldSumOnlyCapturedAccepted() {
        when(capturedEntry.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(capturedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(capturedEntry.getAmount()).thenReturn(new BigDecimal(FIFTY));

        when(transaction.getEntries()).thenReturn(Collections.singletonList(capturedEntry));

        BigDecimal result = PayPalBackofficeUtil.calculateCapturedAmount(Collections.singletonList(transaction));

        assertEquals(new BigDecimal(FIFTY), result);
    }

    @Test
    public void shouldSumAllPlannedAmounts() {
        when(transaction.getPlannedAmount()).thenReturn(new BigDecimal(THIRTY));
        when(transactionTwo.getPlannedAmount()).thenReturn(new BigDecimal(SEVENTY));

        List<PaymentTransactionModel> transactions = Arrays.asList(transaction, transactionTwo);

        BigDecimal result = PayPalBackofficeUtil.calculateTransactionsPlanedAmount(transactions);

        assertEquals(new BigDecimal(HUNDRED), result);
    }

    @Test
    public void shouldSumOnlyAuthorizedAccepted() {
        when(authorizedEntry.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(authorizedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(authorizedEntry.getAmount()).thenReturn(new BigDecimal(HUNDRED));

        when(transaction.getEntries()).thenReturn(Collections.singletonList(authorizedEntry));

        BigDecimal result = PayPalBackofficeUtil.calculateAuthorizedAmount(Collections.singletonList(transaction));

        assertEquals(new BigDecimal(HUNDRED), result);
    }

    @Test
    public void shouldReturnTrueIfConsignmentShippedAndNotTracked() {
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.SHIPPED);
        when(consignmentModel.getTrackingID()).thenReturn(StringUtils.EMPTY);
        when(consignmentModel.getCarrier()).thenReturn(StringUtils.EMPTY);

        assertTrue(PayPalBackofficeUtil.hasShippedNotTrackedConsignments(consignmentModel));
    }

    @Test
    public void shouldReturnFalseIfConsignmentNull() {
        assertFalse(PayPalBackofficeUtil.hasShippedNotTrackedConsignments(null));

    }
    @Test
    public void shouldReturnFalseIfConsighnmentStatusNotShipped(){
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.WAITING);
        when(consignmentModel.getTrackingID()).thenReturn(StringUtils.EMPTY);
        when(consignmentModel.getCarrier()).thenReturn(StringUtils.EMPTY);
        assertFalse(PayPalBackofficeUtil.hasShippedNotTrackedConsignments(consignmentModel));
    }

    @Test
    public void shouldReturnFalseIfTrackingInfoAlreadyProvided(){
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.SHIPPED);
        when(consignmentModel.getTrackingID()).thenReturn(FIFTY);
        when(consignmentModel.getCarrier()).thenReturn(FIFTY);
        assertFalse(PayPalBackofficeUtil.hasShippedNotTrackedConsignments(consignmentModel));
    }

    @After
    public void close() throws Exception{
        closeable.close();
    }

}