package com.paypal.hybris.backoffice.handlers;

import com.hybris.backoffice.widgets.notificationarea.event.NotificationEvent;
import com.hybris.cockpitng.dataaccess.facades.object.ObjectFacade;
import com.hybris.cockpitng.dataaccess.facades.object.exceptions.ObjectSavingException;
import com.hybris.cockpitng.engine.WidgetInstanceManager;
import com.hybris.cockpitng.util.notifications.NotificationService;
import com.paypal.hybris.core.enums.SavedPaymentMethodsLimit;
import com.paypal.hybris.core.model.PayPalConfigurationModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.servicelayer.session.SessionService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalConfigurationEditorAreaLogicHandlerTest {

    private static final String PAY_PAL_CONFIGURATION_DISABLED = "payPalConfigurationDisabled";
    private static final String SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE = "savedPayPalAccountsLimitBefore";
    private static final String PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE_POPUP =
            "payPalSavedAccountsLimitConfigurationMoreThanOnePopup";
    private static final String PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE =
            "payPalSavedAccountsLimitConfigurationMoreThanOne";

    @Mock
    private ObjectFacade objectFacade;

    @Mock
    private NotificationService notificationService;

    @Mock
    private WidgetInstanceManager widgetInstanceManager;

    @Mock
    private PayPalConfigurationModel currentObject;

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private SessionService sessionService;

    @InjectMocks
    private PayPalConfigurationEditorAreaLogicHandler unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(sessionService.getCurrentSession().getAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE))
                .thenReturn(null);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.ONE);
    }

    @Test
    public void shouldShowInfoNotificationWhenPayPalDisabled() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.FALSE);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService).notifyUser(widgetInstanceManager, PAY_PAL_CONFIGURATION_DISABLED, NotificationEvent.Level.INFO);
    }

    @Test
    public void shouldNotShowInfoNotificationWhenPayPalEnabled() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager, PAY_PAL_CONFIGURATION_DISABLED, NotificationEvent.Level.INFO);
    }

    @Test
    public void shouldShowWarningNotificationWhenPayPalDisabled() {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.FALSE);

        unit.beforeEditorAreaRender(widgetInstanceManager, currentObject);

        verify(notificationService).notifyUser(widgetInstanceManager, PAY_PAL_CONFIGURATION_DISABLED, NotificationEvent.Level.WARNING);
    }

    @Test
    public void shouldNotShowWarningNotificationWhenPayPalEnabled() {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);

        unit.beforeEditorAreaRender(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager, PAY_PAL_CONFIGURATION_DISABLED, NotificationEvent.Level.WARNING);
    }

    @Test
    public void shouldAlwaysWarnUserInEditorAreaWhenSavedAccountsLimitIsMoreThanOne() {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.TWO);

        unit.beforeEditorAreaRender(widgetInstanceManager, currentObject);

        verify(notificationService).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE_POPUP, NotificationEvent.Level.WARNING);
        verify(sessionService.getCurrentSession()).setAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE, SavedPaymentMethodsLimit.TWO);
    }

    @Test
    public void shouldNotWarnUserInEditorAreaWhenSavedAccountsLimitIsOne() {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.ONE);

        unit.beforeEditorAreaRender(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE_POPUP, NotificationEvent.Level.WARNING);
        verify(sessionService.getCurrentSession()).setAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE, SavedPaymentMethodsLimit.ONE);
    }

    @Test
    public void shouldNotWarnUserInEditorAreaWhenPayPalDisabledAndSavedAccountsLimitIsMoreThanOne() {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.FALSE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);

        unit.beforeEditorAreaRender(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE_POPUP, NotificationEvent.Level.WARNING);
        verify(sessionService.getCurrentSession()).setAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE, SavedPaymentMethodsLimit.THREE);
    }

    @Test
    public void shouldNotifyUserInGlobalMessageWhenSavedAccountsLimitWasOneAndChangedForMoreThanOne() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);
        when(sessionService.getCurrentSession().getAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE))
                .thenReturn(SavedPaymentMethodsLimit.ONE);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE, NotificationEvent.Level.INFO);
    }

    @Test
    public void shouldNotNotifyUserInGlobalMessageWhenSavedAccountsLimitWasAndChangedForMoreThanOne() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);
        when(sessionService.getCurrentSession().getAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE))
                .thenReturn(SavedPaymentMethodsLimit.TWO);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE, NotificationEvent.Level.INFO);
    }

    @Test
    public void shouldNotNotifyUserInGlobalMessageWhenSavedAccountsLimitBeforeIsNull() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.TRUE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);
        when(sessionService.getCurrentSession().getAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE))
                .thenReturn(null);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE, NotificationEvent.Level.INFO);
    }

    @Test
    public void shouldNotNotifyUserInGlobalMessageWhenPayPalIsDisabledAndSavedAccountsLimitWasOneAndChangedForMoreThanOne() throws ObjectSavingException {
        when(currentObject.isPayPalEnable()).thenReturn(Boolean.FALSE);
        when(currentObject.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);
        when(sessionService.getCurrentSession().getAttribute(SAVED_PAY_PAL_ACCOUNTS_LIMIT_VALUE_BEFORE))
                .thenReturn(SavedPaymentMethodsLimit.ONE);

        unit.performSave(widgetInstanceManager, currentObject);

        verify(notificationService, never()).notifyUser(widgetInstanceManager,
                PAYPAL_SAVED_ACCOUNTS_LIMIT_CONFIGURATION_MORE_THAN_ONE, NotificationEvent.Level.INFO);
    }

}
