package com.paypal.hybris.backoffice.action.order.voidorder;

import com.hybris.cockpitng.actions.ActionContext;
import com.hybris.cockpitng.actions.ActionResult;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.ordercancel.CancelDecision;
import de.hybris.platform.ordercancel.OrderCancelService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Collections;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalVoidOrderActionTest {

    private static final String SOCKET_OUT_CONTEXT = "cancelOrderContext";

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private UserService userService;

    @Mock
    private OrderCancelService orderCancelService;

    @Mock
    private AbstractOrderEntryModel orderEntryModel;

    @Mock
    private List<OrderStatus> notCancellableOrderStatus;

    @Mock
    private ActionContext<OrderModel> actionContext;

    @Mock
    private OrderModel orderModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private PaymentTransactionEntryModel transactionEntryModel;

    @Mock
    private CancelDecision cancelDecision;

    @Mock
    private CustomerModel customerModel;

    @Spy
    @InjectMocks
    private PayPalVoidOrderAction unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(actionContext.getData()).thenReturn(orderModel);
    }

    @Test
    public void shouldPerformSuccessfully() {
        assertEquals(ActionResult.SUCCESS, unit.perform(actionContext).getResultCode());

        verify(unit).sendOutput(SOCKET_OUT_CONTEXT, orderModel);
    }


    @Test
    public void shouldReturnTrueWhenCanPerform() {
        when(orderModel.getEntries()).thenReturn(Collections.singletonList(orderEntryModel));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(orderCancelService.isCancelPossible(orderModel, customerModel, Boolean.TRUE, Boolean.TRUE)).thenReturn(cancelDecision);
        when(cancelDecision.isAllowed()).thenReturn(Boolean.TRUE);
        when(orderModel.getStatus()).thenReturn(OrderStatus.APPROVED);
        when(notCancellableOrderStatus.contains(OrderStatus.APPROVED)).thenReturn(Boolean.FALSE);

        when(orderModel.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderNull() {
        when(actionContext.getData()).thenReturn(null);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenNotEntries() {
        when(orderModel.getEntries()).thenReturn(Collections.emptyList());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenCancelNotPossible() {
        when(orderModel.getEntries()).thenReturn(Collections.singletonList(orderEntryModel));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(orderCancelService.isCancelPossible(orderModel, customerModel, Boolean.TRUE, Boolean.TRUE)).thenReturn(cancelDecision);
        when(cancelDecision.isAllowed()).thenReturn(Boolean.FALSE);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenNotCancellableOrderStatusContainActualOrderStatus() {
        when(orderModel.getEntries()).thenReturn(Collections.singletonList(orderEntryModel));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(orderCancelService.isCancelPossible(orderModel, customerModel, Boolean.TRUE, Boolean.TRUE)).thenReturn(cancelDecision);
        when(cancelDecision.isAllowed()).thenReturn(Boolean.TRUE);
        when(orderModel.getStatus()).thenReturn(OrderStatus.REJECTED);
        when(notCancellableOrderStatus.contains(OrderStatus.REJECTED)).thenReturn(Boolean.TRUE);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenNotAuthorized() {
        when(orderModel.getEntries()).thenReturn(Collections.singletonList(orderEntryModel));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(orderCancelService.isCancelPossible(orderModel, customerModel, Boolean.TRUE, Boolean.TRUE)).thenReturn(cancelDecision);
        when(cancelDecision.isAllowed()).thenReturn(Boolean.TRUE);
        when(orderModel.getStatus()).thenReturn(OrderStatus.APPROVED);
        when(notCancellableOrderStatus.contains(OrderStatus.APPROVED)).thenReturn(Boolean.FALSE);

        when(orderModel.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenCaptured() {
        when(orderModel.getEntries()).thenReturn(Collections.singletonList(orderEntryModel));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(orderCancelService.isCancelPossible(orderModel, customerModel, Boolean.TRUE, Boolean.TRUE)).thenReturn(cancelDecision);
        when(cancelDecision.isAllowed()).thenReturn(Boolean.TRUE);
        when(orderModel.getStatus()).thenReturn(OrderStatus.APPROVED);
        when(notCancellableOrderStatus.contains(OrderStatus.APPROVED)).thenReturn(Boolean.FALSE);

        when(orderModel.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenNeedsConfirmation() {
        assertFalse(unit.needsConfirmation(actionContext));
    }

    @Test
    public void shouldReturnNullWhenGetConfirmationMessage() {
        assertNull(unit.getConfirmationMessage(actionContext));
    }
}