package com.paypal.hybris.backoffice.action.order.tracking;

import com.hybris.cockpitng.actions.ActionContext;
import com.paypal.hybris.core.enums.PaymentStatusType;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.enums.ConsignmentStatus;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.Collections;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalSendTrackingInfoActionTest {

    private AutoCloseable closeable;

    @Mock
    private OrderModel orderModel;
    @Mock
    private ActionContext<OrderModel> actionContext;
    @Mock
    private ConsignmentModel consignmentModel;

    @InjectMocks
    private PayPalSendTrackingInfoAction unit;

    @Before
    public void setup() {
        closeable = MockitoAnnotations.openMocks(this);

        when(actionContext.getData()).thenReturn(orderModel);
        when(orderModel.getConsignments()).thenReturn(Collections.singleton(consignmentModel));
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.SHIPPED);
        when(orderModel.getStatus()).thenReturn(OrderStatus.COMPLETED);
        when(orderModel.getPaymentStatusType()).thenReturn(PaymentStatusType.COMPLETED);
    }

    @Test
    public void shouldReturnFalseWhenOrderNull() {
        when(actionContext.getData()).thenReturn(null);
        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenNoShippedConsignmentsInOrder() {
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.WAITING);
        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenOrderStatusNotCaptured() {
        when(orderModel.getStatus()).thenReturn(OrderStatus.ON_VALIDATION);
        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenNoConsignmentsInOrder() {
        when(orderModel.getConsignments()).thenReturn(null);
        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenConsignmentInOrderIsNull() {
        ConsignmentModel nullConsignment = null;
        when(orderModel.getConsignments()).thenReturn(Collections.singleton(nullConsignment));
        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnTrueWhenOrderCapturedAndConsignmentShipped() {
        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenPaymentStatusTypeRefunded() {
        when(orderModel.getPaymentStatusType()).thenReturn(PaymentStatusType.REFUNDED);
        assertFalse(unit.canPerform(actionContext));
    }

    @After
    public void close() throws Exception {
        closeable.close();
    }

}
