package com.paypal.hybris.backoffice.action.order.refund;

import com.hybris.cockpitng.actions.ActionContext;
import com.hybris.cockpitng.actions.ActionResult;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.math.BigDecimal;
import java.util.Collections;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalRefundActionTest {

    private static final String SOCKET_OUT_CONTEXT = "paypalPartialRefundContext";
    private static final String SUCCESSFUL = "SUCCESSFUL";
    private static final String NOT_SUCCESSFUL = "NOT_SUCCESSFUL";
    private static final int ELEVEN = 11;

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private OrderModel order;

    @Mock
    private PaymentTransactionEntryModel transactionEntry;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private ActionContext<OrderModel> actionContext;

    @Spy
    @InjectMocks
    private PayPalRefundAction unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(actionContext.getData()).thenReturn(order);
    }

    @Test
    public void shouldPerformSuccessfully() {
        assertEquals(ActionResult.SUCCESS, unit.perform(actionContext).getResultCode());

        verify(unit).sendOutput(SOCKET_OUT_CONTEXT, order);
    }

    @Test
    public void shouldReturnTrue() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntry));

        when(transactionEntry.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL.name());
        when(transactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        when(order.getStatus()).thenReturn(OrderStatus.PAYMENT_CAPTURED);

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderNull() {
        when(actionContext.getData()).thenReturn(null);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderHasNotPaymentTransactions() {
        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenTransactionsHasNotEntries() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.emptyList());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenTransactionStatusDetailsNotSuccessful() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntry));

        when(transactionEntry.getTransactionStatusDetails()).thenReturn(NOT_SUCCESSFUL);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenTransactionStatusRejected() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntry));

        when(transactionEntry.getTransactionStatusDetails()).thenReturn(SUCCESSFUL);
        when(transactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenCaptureAmountZero() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntry));

        when(transactionEntry.getTransactionStatusDetails()).thenReturn(SUCCESSFUL);
        when(transactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.CANCEL);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderStatusCancelled() {
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntry));

        when(transactionEntry.getTransactionStatusDetails()).thenReturn(SUCCESSFUL);
        when(transactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        when(order.getStatus()).thenReturn(OrderStatus.CANCELLED);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnElevenWhenTransactionTypeCapture() {
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        assertEquals(new BigDecimal(ELEVEN), unit.addCapturedAmount(BigDecimal.ONE, transactionEntry));
    }

    @Test
    public void shouldReturnElevenWhenTransactionTypePartialCapture() {
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        assertEquals(new BigDecimal(ELEVEN), unit.addCapturedAmount(BigDecimal.ONE, transactionEntry));
    }

    @Test
    public void shouldReturnZeroWhenTransactionTypeRefundPartial() {
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.REFUND_PARTIAL);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        assertEquals(BigDecimal.ZERO, unit.addCapturedAmount(BigDecimal.TEN, transactionEntry));
    }

    @Test
    public void shouldReturnZeroWhenTransactionTypeRefundStandalone() {
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.REFUND_STANDALONE);
        when(transactionEntry.getAmount()).thenReturn(BigDecimal.TEN);

        assertEquals(BigDecimal.ZERO, unit.addCapturedAmount(BigDecimal.TEN, transactionEntry));
    }

    @Test
    public void shouldReturnZeroWhenTransactionTypeCancel() {
        when(transactionEntry.getType()).thenReturn(PaymentTransactionType.CANCEL);

        assertEquals(BigDecimal.ZERO, unit.addCapturedAmount(BigDecimal.ZERO, transactionEntry));
    }
}