package com.paypal.hybris.backoffice.action.order.reauthorize;

import com.hybris.backoffice.widgets.notificationarea.NotificationService;
import com.hybris.cockpitng.actions.ActionContext;
import com.hybris.cockpitng.actions.ActionResult;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.store.BaseStoreModel;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Calendar;
import java.util.Collections;
import java.util.Date;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalReauthorizeActionTest {

    private static final String MANUAL_PAYMENT_REAUTH_EVENT = "ManualPaymentReauthEvent";

    private static final String SOCKET_OUT_CONTEXT = "paypalReauthorizeContext";

    private static final int MIN_DAYS = 4;

    private static final int MAX_DAYS = 29;

    private static final String ORDER_CODE = "orderCode";

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private BusinessProcessService businessProcessService;

    @Mock
    private ModelService modelService;

    @Mock
    private OrderModel order;

    @Mock
    private OrderProcessModel orderProcessModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel cardPaymentInfoModel;

    @Mock
    private PaymentInfoModel paymentInfoModel;

    @Mock
    private NotificationService notificationService;

    @Mock
    private BaseStoreModel baseStoreModel;

    @Mock
    private ActionContext<OrderModel> actionContext;

    @Spy
    @InjectMocks
    private PayPalReauthorizeAction unit;

    private Date createOrderDate;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(actionContext.getData()).thenReturn(order);
        createOrderDate = new Date();
    }

    @Test
    public void shouldPerformSuccessfully() {
        when(order.getOrderProcess()).thenReturn(Collections.singletonList(orderProcessModel));
        when(orderProcessModel.getCode()).thenReturn(ORDER_CODE);
        when(order.getStore()).thenReturn(baseStoreModel);
        when(baseStoreModel.getSubmitOrderProcessCode()).thenReturn(ORDER_CODE);

        assertEquals(ActionResult.SUCCESS, unit.perform(actionContext).getResultCode());

        verify(unit).sendOutput(SOCKET_OUT_CONTEXT, order);
        verify(businessProcessService).triggerEvent(ORDER_CODE + "_" + MANUAL_PAYMENT_REAUTH_EVENT);
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderNull() {
        when(actionContext.getData()).thenReturn(null);

        assertFalse(unit.canPerform(actionContext));
    }


    @Test
    public void shouldReturnTrueWhenCanPerformWhenPaymentCard() {
        Date createdOrderDate = getDateWithOffset(-15);

        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);

        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        when(order.getStatus()).thenReturn(OrderStatus.PAYMENT_NOT_CAPTURED);
        when(order.getDate()).thenReturn(createdOrderDate);

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnTrueWhenCanPerformWhenPaymentNotCard() {
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(order.getStatus()).thenReturn(OrderStatus.PAYMENT_NOT_CAPTURED);

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderFlowActive() {
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);

        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderRejected() {
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);

        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        when(order.getStatus()).thenReturn(OrderStatus.REJECTED);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenOrderNotApplicableForReauthorization() {
        Date createdOrderDate = getDateWithOffset(-3);

        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);

        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        when(order.getStatus()).thenReturn(OrderStatus.PAYMENT_NOT_CAPTURED);
        when(order.getDate()).thenReturn(createdOrderDate);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenCanNotPerformWhenPaymentInfoIsNotCardAndOrderStatusRejected() {
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(order.getStatus()).thenReturn(OrderStatus.REJECTED);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenNeedsConfirmation() {
        assertFalse(unit.needsConfirmation(actionContext));
    }

    @Test
    public void shouldReturnNullWhenGetConfirmationMessage() {
        assertNull(unit.getConfirmationMessage(actionContext));
    }

    @Test
    public void shouldBeApplicableForReauthorization() {
        createOrderDate.setTime(new Date().getTime());
    }


    @Test
    public void shouldReturnFalseWhenOrderDateBeforeMinDays() {
        Date createdOrderDate = getDateWithOffset(-3);
        assertFalse(unit.isOrderApplicableForReauthorization(createdOrderDate));
    }

    @Test
    public void shouldReturnTrueWhenOrderDateOnMinDays() {
        Date createdOrderDate = getDateWithOffset(-MIN_DAYS);
        assertTrue(unit.isOrderApplicableForReauthorization(createdOrderDate));
    }

    @Test
    public void shouldReturnTrueWhenOrderDateBetweenMinAndMaxDays() {
        Date createdOrderDate = getDateWithOffset(-15);
        assertTrue(unit.isOrderApplicableForReauthorization(createdOrderDate));
    }

    @Test
    public void shouldReturnFalseWhenOrderDateOnMaxDays() {
        Date createdOrderDate = getDateWithOffset(-MAX_DAYS);
        assertFalse(unit.isOrderApplicableForReauthorization(createdOrderDate));
    }

    @Test
    public void shouldReturnFalseWhenOrderDateAfterMaxDays() {
        Date createdOrderDate = getDateWithOffset(-30);
        assertFalse(unit.isOrderApplicableForReauthorization(createdOrderDate));
    }

    private Date getDateWithOffset(int daysOffset) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_YEAR, daysOffset);
        return calendar.getTime();
    }

}