package com.paypal.hybris.backoffice.action.order.authorize;

import com.hybris.cockpitng.actions.ActionContext;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.session.SessionExecutionBody;
import de.hybris.platform.servicelayer.session.SessionService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalAuthorizeOrderActionTest {

    private static final String SOCKET_OUT_CONTEXT = "paypalAuthorizeOrderContext";

    private static final String SUCCESS = "success";

    @Mock
    private ActionContext<OrderModel> actionContext;

    @Mock
    private OrderModel order;

    @Mock
    private PayPalCreditCardPaymentInfoModel cardPaymentInfoModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private PaymentTransactionEntryModel transactionEntryModel;

    @Mock
    private PaymentInfoModel paymentInfoModel;

    @Mock
    private BaseSiteModel baseSiteModel;

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private PayPalConfigurationService payPalConfigurationService;

    @Mock
    private SessionService sessionService;

    @Spy
    @InjectMocks
    private PayPalAuthorizeOrderAction unit;


    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldReturnSuccessActionResult() {
        when(actionContext.getData()).thenReturn(order);

        assertEquals(unit.perform(actionContext).getResultCode(), SUCCESS);
        verify(unit).sendOutput(SOCKET_OUT_CONTEXT, order);
    }

    @Test
    public void shouldReturnTrue() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);
        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(order.getStatus()).thenReturn(OrderStatus.APPROVED_BY_MERCHANT);

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.TEN);

        when(order.getTotalPrice()).thenReturn(120.3);

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenAuthorizationIsNotAvailableForOrder() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);
        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(order.getStatus()).thenReturn(OrderStatus.APPROVED_BY_MERCHANT);

        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.ZERO);
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.ZERO);

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertFalse(unit.canPerform(actionContext));
    }


    @Test
    public void shouldReturnFalseWhenOrderStatusCancelled() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);
        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(order.getStatus()).thenReturn(OrderStatus.CANCELLED);

        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnTrueWhenNotPaymentTransactions() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);
        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(order.getStatus()).thenReturn(OrderStatus.APPROVED_BY_MERCHANT);
        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());

        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertTrue(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalse() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenOrderNull() {
        when(actionContext.getData()).thenReturn(null);

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseIfPaymentInfoNotPayPalCreditCardPaymentInfoModel() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertFalse(unit.canPerform(actionContext));
    }

    @Test
    public void shouldReturnFalseWhenSaveOrderFlowNotActive() {
        when(actionContext.getData()).thenReturn(order);
        when(order.getSite()).thenReturn(baseSiteModel);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(order.getPaymentInfo()).thenReturn(cardPaymentInfoModel);
        when(cardPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        when(sessionService.executeInLocalViewWithParams(any(), any())).thenAnswer(new Answer<Object>() {
            @Override
            public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                return ((SessionExecutionBody) invocationOnMock.getArguments()[1]).execute();
            }
        });

        assertFalse(unit.canPerform(actionContext));
    }
}
