package com.paypal.hybris.backoffice.widgets.order.tracking;

import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.core.events.CockpitEventQueue;
import com.hybris.cockpitng.core.events.impl.DefaultCockpitEvent;
import com.hybris.cockpitng.dataaccess.facades.object.ObjectCRUDHandler;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.backoffice.util.PayPalBackofficeUtil;
import com.paypal.hybris.core.model.PayPalCarrierModel;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCarrierService;
import com.paypal.hybris.core.service.PayPalTrackingInfoService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionExecutionBody;
import de.hybris.platform.servicelayer.session.SessionService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;
import org.zkoss.zul.impl.InputElement;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Map;
import java.util.Optional;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.TRACKING_ORDER_EMPTY_FIELD_ERROR;

public class PayPalSendTrackingInfoController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalSendTrackingInfoController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String CURRENT_SITE = "currentSite";
    private static final String ORDER_TRACKING_TRACKINGINFOS = "paypalbackoffice.order.tracking.trackinginfos";
    private static final String ORDER_TRACKING_TITLE = "paypalbackoffice.order.tracking.title";
    private static final String TRACKING_SUCCESS_MESSAGE = "paypalbackoffice.order.tracking.success.message";
    private static final String TRACKING_ERROR_MESSAGE = "paypalbackoffice.order.tracking.error.message";
    private static final String TRANSACTION_ID_WITH_AMOUNT = "%s - %s%.2f";

    @Resource
    private PayPalCarrierService payPalCarrierService;
    @Resource
    private ModelService modelService;
    @Resource
    private CockpitEventQueue cockpitEventQueue;
    @Resource
    private PayPalTrackingInfoService payPalTrackingInfoService;
    @Resource
    private SessionService sessionService;
    @WireVariable
    private transient PaymentTransactionsService payPalPaymentTransactionsService;
    @Wire
    private Combobox consignmentsComboBox;
    @Wire
    private Combobox captureTransactionsComboBox;
    @Wire
    private Combobox carriersComboBox;
    @Wire
    private Textbox trackingIdTextBox;

    private OrderModel order;

    @SocketEvent(socketId = IN_SOCKET)
    public void initWidget(OrderModel inputOrder) {
        this.order = inputOrder;
        this.getWidgetInstanceManager()
                .setTitle(this.getWidgetInstanceManager().getLabel(ORDER_TRACKING_TRACKINGINFOS));
        createConsignmentsIdDropDown(this.order);
        createCaptureTransactionsIdDropDown(this.order);
        createCarrierDropDown();
    }

    @ViewEvent(
            componentID = "confirm",
            eventName = "onClick"
    )
    public void confirmTrackingForConsignment() {
        validateInput();
        String consignmentCode = this.consignmentsComboBox.getValue();

        this.order.getConsignments().stream()
                .filter(c -> consignmentCode.equals(c.getCode()))
                .findFirst()
                .ifPresentOrElse(this::saveTrackingInfo, this::showSavingError);
        final DefaultCockpitEvent event = new DefaultCockpitEvent(ObjectCRUDHandler.OBJECTS_UPDATED_EVENT, order.getConsignments(), null);
        cockpitEventQueue.publishEvent(event);
        sendOutput(OUT_MODIFIED_ITEM, order);
    }

    private void saveTrackingInfo(ConsignmentModel consignmentModel) {
        String trackingNumber = this.trackingIdTextBox.getValue();
        PayPalCarrierModel carrierModel = this.carriersComboBox.getSelectedItem().getValue();
        PaymentTransactionEntryModel selectedTransaction = this.captureTransactionsComboBox.getSelectedItem().getValue();

        try {
            String resultStatus = sessionService.executeInLocalViewWithParams(
                    Map.of(CURRENT_SITE, order.getSite()), new SessionExecutionBody() {
                        @Override
                        public String execute() {
                            return payPalTrackingInfoService.sendTrackingInfoToPayPalForConsignment(selectedTransaction,
                                    consignmentModel, carrierModel.getCode(), trackingNumber);
                        }
                    });

            if (resultStatus != null) {
                Messagebox.show(getLabel(TRACKING_SUCCESS_MESSAGE), getLabel(ORDER_TRACKING_TITLE), Messagebox.OK,
                        Messagebox.ERROR);
                consignmentModel.setCarrier(carrierModel.getName());
                consignmentModel.setTrackingID(trackingNumber);
                modelService.save(consignmentModel);
            } else {
                showSavingError();
            }
        } catch (Exception e) {
            LOG.error(e.getMessage());
            showSavingError();
        }
    }

    protected void validateInput() {
        checkIfEmpty(consignmentsComboBox);
        checkIfEmpty(captureTransactionsComboBox);
        checkIfEmpty(carriersComboBox);
        checkIfEmpty(trackingIdTextBox);
    }

    protected void checkIfEmpty(Textbox textbox) {
        String trackingNumber = textbox.getValue();
        if (StringUtils.isBlank(trackingNumber)) {
            throw new WrongValueException(textbox, getLabel(TRACKING_ORDER_EMPTY_FIELD_ERROR));
        }
    }

    private void showSavingError() {
        Messagebox.show(getLabel(TRACKING_ERROR_MESSAGE), getLabel(ORDER_TRACKING_TITLE), Messagebox.OK,
                Messagebox.ERROR);
    }

    private void createConsignmentsIdDropDown(OrderModel order) {
        for (ConsignmentModel consignment : order.getConsignments()) {
            if (PayPalBackofficeUtil.hasShippedNotTrackedConsignments(consignment)) {
                consignmentsComboBox.appendItem(consignment.getCode());
            }
        }
        consignmentsComboBox.setSelectedIndex(0);
    }

    private void createCarrierDropDown() {
        payPalCarrierService.getAllCarriers().forEach(carrier ->
                carriersComboBox.appendChild(createComboItem(carrier.getName(), carrier)));
    }

    private void createCaptureTransactionsIdDropDown(OrderModel orderModel) {
        payPalPaymentTransactionsService.getTransactionsToRefund(orderModel).forEach(entry ->
                captureTransactionsComboBox.appendChild(
                        createComboItem(createTransactionIdWithAmountLabel(entry), entry)));
        captureTransactionsComboBox.setSelectedIndex(0);
    }

    private String createTransactionIdWithAmountLabel(PaymentTransactionEntryModel transactionEntryModel) {
        return TRANSACTION_ID_WITH_AMOUNT.formatted(transactionEntryModel.getRequestId(),
                transactionEntryModel.getCurrency().getSymbol(),
                transactionEntryModel.getAmount());
    }

    private Comboitem createComboItem(String label, Object value) {
        Comboitem comboitem = new Comboitem();
        comboitem.setLabel(label);
        comboitem.setValue(value);

        return comboitem;
    }

}
