package com.paypal.hybris.backoffice.widgets.order.refund;

import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.core.events.CockpitEventQueue;
import com.hybris.cockpitng.core.events.impl.DefaultCockpitEvent;
import com.hybris.cockpitng.dataaccess.facades.object.ObjectCRUDHandler;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.List;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_INCORRECT_AMOUNT_VALUE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_ZERO_AMOUNT;


public class PayPalRefundController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalRefundController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String PARTIAL_REFUND_CONFIRM_TITLE = "paypal.backoffice.partial.refund.confirm.title";
    private static final String ON_CHANGE = "onChange";

    private PaymentTransactionEntryModel transaction;
    private OrderModel order;

    @Wire
    private Textbox amount;
    @Wire
    private Combobox transactionsCombobox;
    @Wire
    private CockpitEventQueue cockpitEventQueue;

    @WireVariable
    private transient PayPalPaymentService paymentService;

    @WireVariable
    private transient ModelService modelService;
    @WireVariable
    private transient PaymentTransactionsService payPalPaymentTransactionsService;

    @SocketEvent(
            socketId = IN_SOCKET
    )
    public void initPartialRefundForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager()
                .setTitle(this.getWidgetInstanceManager().getLabel(PARTIAL_REFUND_CONFIRM_TITLE) + " "
                        + this.getOrder().getCode());
        createTransactionListDropDown(this.order, this.transactionsCombobox);
        transactionsCombobox.setSelectedIndex(0);
        this.amount.setValue(getAmount());
        transactionsCombobox.addEventListener(ON_CHANGE, event ->
                amount.setValue(getAmount()));
    }

    @ViewEvent(
            componentID = "refundrequest",
            eventName = "onClick"
    )
    public void confirmRefund() {
        validateAmount();
        this.transaction = (PaymentTransactionEntryModel) transactionsCombobox
                .getAttribute(transactionsCombobox.getSelectedItem().getLabel());
        try {
            BigDecimal amountAvailableForRefund = getAmountToRefund();
            if (amountAvailableForRefund.compareTo(new BigDecimal(this.amount.getValue())) < 0) {
                showMessageBox(getLabel(REFUND_INCORRECT_AMOUNT_VALUE) + getAmountAvailableForRefund(),
                        getLabel(REFUND_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.ERROR);
                sendOutput(OUT_MODIFIED_ITEM, order);
            } else {
                processRefund();
            }
        } catch (Exception e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            showMessageBox(e.getMessage() != null ? "Refund failed. Error message: " + e.getMessage() : "Refund failed." + getAmountAvailableForRefund(),
                    getLabel(REFUND_ERROR),
                    Messagebox.OK,
                    Messagebox.ERROR);
        }
        DefaultCockpitEvent event = new DefaultCockpitEvent(ObjectCRUDHandler.OBJECTS_UPDATED_EVENT, order, null);
        cockpitEventQueue.publishEvent(event);
    }

    private void processRefund() {
        final PaymentTransactionEntryModel refundResult = paymentService
                .refund(this.transaction, new BigDecimal(this.amount.getValue()));
        if (TransactionStatus.ACCEPTED.toString().equals(refundResult.getTransactionStatus())) {
            showMessageBox(getLabel(REFUND_SUCCESS),
                    getLabel(REFUND_TITLE) + " " + this.getOrder().getCode(),
                    Messagebox.OK,
                    Messagebox.INFORMATION);
            sendOutput(OUT_MODIFIED_ITEM, order);
            setPaymentStatusType();
        } else {
            showMessageBox(refundResult.getTransactionStatusDetails() + getAmountAvailableForRefund(),
                    getLabel(REFUND_TITLE) + " " + this.getOrder().getCode(),
                    Messagebox.OK,
                    Messagebox.ERROR);
        }
    }


    private Combobox createTransactionListDropDown(OrderModel order, final Combobox transactionsCombobox) {
        for (PaymentTransactionModel paymentTransaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel paymentEntry : paymentTransaction.getEntries()) {
                if (TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus()) &&
                        (PaymentTransactionType.CAPTURE.equals(paymentEntry.getType())
                                || PaymentTransactionType.PARTIAL_CAPTURE.equals(paymentEntry.getType()))) {
                    transactionsCombobox.appendItem(paymentEntry.getRequestId());
                    transactionsCombobox.setAttribute(paymentEntry.getRequestId(), paymentEntry);
                }
            }
        }
        return transactionsCombobox;
    }

    private String getAmountAvailableForRefund() {
        return ". \n \n The available amount to refund is " + getAmount() + " " + this.transaction.getCurrency().getIsocode();
    }

    protected String getAmount() {
        BigDecimal amountToRefund = getAmountToRefund();
        if (amountToRefund.compareTo(BigDecimal.ZERO) > 0) {
            return formatAmount(amountToRefund);
        } else {
            return BigDecimal.ZERO.toString();
        }
    }

    private BigDecimal getAmountToRefund() {
        PaymentTransactionEntryModel currentModel = (PaymentTransactionEntryModel) transactionsCombobox
                .getAttribute(transactionsCombobox.getSelectedItem().getLabel());
        return currentModel.getAmount().subtract(payPalPaymentTransactionsService.calculateRefundedAmount(currentModel));
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private void validateAmount() {
        String value = this.amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(REFUND_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel(REFUND_ZERO_AMOUNT));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(REFUND_INVALID_FORMAT_AMOUNT));
        }
    }

    private void setPaymentStatusType() {
        modelService.refresh(order);
        List<PaymentTransactionEntryModel> captureTransactions = payPalPaymentTransactionsService.getTransactionsToRefund(order);
        PaymentStatusType paymentStatusType = captureTransactions.isEmpty() ? PaymentStatusType.REFUNDED : PaymentStatusType.PARTIAL_REFUND;
        order.setPaymentStatusType(paymentStatusType);
        modelService.save(order);
    }

    public void setTransactionsCombobox(Combobox transactionsCombobox) {
        this.transactionsCombobox = transactionsCombobox;
    }

    protected void showMessageBox(String message, String title, int buttons, String icon) {
        Messagebox.show(message, title, buttons, icon);
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

}
