package com.paypal.hybris.backoffice.widgets.order.reauthorize;

import com.google.common.base.Preconditions;
import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.backoffice.util.PayPalBackofficeUtil;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.exception.PayPalReauthorizeAdapterException;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Currency;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ZERO_AMOUNT;

public class PayPalReauthorizeController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalReauthorizeController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String REAUTHORIZATION_EXCEPTION = "Reauthorization exception occurred: %s";
    private static final String ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE = "orderModel can not be null";
    private static final String ORDER_IS_NOT_EXIST = "Order is not exist";

    private OrderModel order;

    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;
    @Wire
    private Combobox transactions;

    @WireVariable
    private transient PayPalPaymentService paymentService;

    @WireVariable
    private transient ModelService modelService;

    @WireVariable
    private BusinessProcessService businessProcessService;

    @WireVariable
    private CommandFactoryRegistry commandFactoryRegistry;

    @SocketEvent(socketId = IN_SOCKET)
    public void initReauthorizeRequestForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager()
            .setTitle(this.getWidgetInstanceManager().getLabel(REAUTHORIZE_TITLE)
                + " " + this.getOrder().getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        this.amount.setValue(formatAmount(getPossibleAmountForReauthorize(inputOrder)));
        configureTransactionsCombobox(inputOrder);
    }

    private void configureTransactionsCombobox(final OrderModel inputOrder) {
        List<PaymentTransactionEntryModel> trans = getAuthorizeTransactions(inputOrder);
        LOG.info("Authorization transactions: " + trans);

        for (PaymentTransactionEntryModel v : trans) {
            Comboitem ci = new Comboitem();
            ci.setValue(v);
            ci.setLabel(v.getRequestId());

            transactions.appendChild(ci);
        }

        if (!transactions.getItems().isEmpty()) {
            transactions.setSelectedItem(transactions.getItems().stream().findFirst().get());
        }
    }

    @ViewEvent(componentID = "transactions", eventName = "onSelect")
    public void selectTransactionEntry() {
        Comboitem ci = transactions.getSelectedItem();
        amount.setValue(formatAmount(((PaymentTransactionEntryModel) ci.getValue()).getAmount()));
    }

    @ViewEvent(componentID = "reauthorizerequest", eventName = "onClick")
    public void confirm() {
        PaymentTransactionEntryModel selectedEntry = new PaymentTransactionEntryModel();
        if (transactions.getSelectedIndex() != -1) {
            selectedEntry = transactions.getSelectedItem().getValue();
        }
        validateAmount();
        getTransactionInfo(selectedEntry);
    }

    private void getTransactionInfo(PaymentTransactionEntryModel selectedEntry) {
        final BigDecimal transactionAmount = new BigDecimal(this.amount.getValue());

        try {
            Boolean isComplete = reauthorize(selectedEntry, transactionAmount);
            if (isComplete) {
                order.setAlreadyReauthorized(Boolean.TRUE);
                modelService.save(order);
                showMessageBox(getLabel(REAUTHORIZE_SUCCESS),
                        getLabel(REAUTHORIZE_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.INFORMATION);
                sendOutput(OUT_MODIFIED_ITEM, order);
            } else {
                showMessageBox(ORDER_IS_NOT_EXIST,
                            getLabel(REAUTHORIZE_ERROR),
                            Messagebox.OK,
                            Messagebox.ERROR);
            }
        } catch (Exception e) {
            order.setPaymentStatusType(PaymentStatusType.FAILED);
            modelService.save(order);
            showErrorMessage(e);
        }
    }

    protected Boolean reauthorize(final PaymentTransactionEntryModel selectedEntry, final BigDecimal amount) {
        if (order == null) {
            return Boolean.FALSE;
        }
        modelService.refresh(getOrder());
        SubscriptionAuthorizationRequest request = prepareReauthorizeRequest(amount, selectedEntry);
        PaymentTransactionModel paymentTransactionModel = selectedEntry.getPaymentTransaction();
        try {
            PayPalAuthorizationResult authorizationResult = (PayPalAuthorizationResult) paymentService.reauthorize(request);

            order.setPaymentStatusType(PaymentStatusType.PENDING);
            updateAuthorizeTransactionEntry(selectedEntry, amount, authorizationResult);
            modelService.saveAll(order, selectedEntry);
            modelService.refresh(paymentTransactionModel);

            paymentTransactionModel.setPlannedAmount(PayPalBackofficeUtil.calculateAuthorizedAmount(
                    getOrder().getPaymentTransactions()));
            paymentTransactionModel.setRequestId(authorizationResult.getAuthorizationCode());
            modelService.save(paymentTransactionModel);
            modelService.saveAll();
        } catch (PayPalReauthorizeAdapterException e) {
            createFailedReauthorizationTransactionEntryModel(e, request, paymentTransactionModel);
            throw new AdapterException(e.getMessage());
        }

        return Boolean.TRUE;
    }

    private void updateAuthorizeTransactionEntry(PaymentTransactionEntryModel selectedEntry, BigDecimal amount,
                                                 PayPalAuthorizationResult authorizationResult) {
        selectedEntry.setAmount(amount);
        selectedEntry.setRequestId(authorizationResult.getAuthorizationCode());
        selectedEntry.setDebugId(authorizationResult.getDebugId());
        selectedEntry.setRequest(authorizationResult.getRequestField());
        selectedEntry.setResponse(authorizationResult.getResponseField());
        selectedEntry.setUpdateTime(authorizationResult.getUpdateTime());
        selectedEntry.setExpirationTime(authorizationResult.getExpirationTime());
    }

    protected void createFailedReauthorizationTransactionEntryModel(Exception e, SubscriptionAuthorizationRequest request,
                                                                 PaymentTransactionModel paymentTransactionModel) {
        PaymentTransactionEntryModel entry = new PaymentTransactionEntryModel();
        entry.setType(PaymentTransactionType.AUTHORIZATION);
        entry.setTime(new Date());
        entry.setPaymentTransaction(paymentTransactionModel);
        entry.setTransactionStatus(TransactionStatus.ERROR.name());
        entry.setTransactionStatus(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name());
        entry.setCode(PayPalCommandsUtil.getNewPaymentTransactionEntryCode(paymentTransactionModel,
                PaymentTransactionType.AUTHORIZATION));
        entry.setFailureReason(e.getMessage());
        entry.setRequest(PayPalCommandsUtil.getValueAsString(request));
        if (e instanceof PayPalReauthorizeAdapterException exception) {
            entry.setResponse(exception.getParentException().getMessage());
            entry.setRequest(PayPalCommandsUtil.getValueAsString(exception.getRequest()).replaceAll("\\s", StringUtils.EMPTY));
            entry.setDebugId(exception.getDebugId());
            entry.setUpdateTime(new Date());
        }

        PayPalCommandsUtil.saveTransactionEntryToTransaction(modelService, paymentTransactionModel, entry);
    }

    protected SubscriptionAuthorizationRequest prepareReauthorizeRequest(final BigDecimal amount,
        final PaymentTransactionEntryModel selectedEntry) {
        final String merchantTransactionCode = selectedEntry.getPaymentTransaction().getCode();
        final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
        final String paymentProvider = selectedEntry.getPaymentTransaction().getPaymentProvider();
        return new SubscriptionAuthorizationRequest(merchantTransactionCode,
            selectedEntry.getRequestId(), currency, amount, null, paymentProvider);
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private void validateAmount() {
        String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(REAUTHORIZE_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel(REAUTHORIZE_ZERO_AMOUNT));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(REAUTHORIZE_INVALID_FORMAT_AMOUNT));
        }
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

    private List<PaymentTransactionEntryModel> getAuthorizeTransactions(final OrderModel order) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(paymentEntry -> (PaymentTransactionType.AUTHORIZATION.equals(paymentEntry.getType())
                && TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus())))
            .collect(Collectors.toList());

    }

    private void showErrorMessage(Exception e) {
        String message = REAUTHORIZATION_EXCEPTION.formatted(e);
        LOG.error(message, e);
        showMessageBox(e.getMessage(), getLabel(REAUTHORIZE_ERROR), Messagebox.OK, Messagebox.ERROR);
    }

    protected void showMessageBox(String message, String title, int buttons, String icon) {
        Messagebox.show(message, title, buttons, icon);
    }

    private BigDecimal getPossibleAmountForReauthorize(OrderModel orderModel) {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);
        return BigDecimal.valueOf(orderModel.getTotalPrice());
    }

    private void validateParameterNotNull(Object parameter, String nullMessage) {
        Preconditions.checkArgument(parameter != null, nullMessage);
    }

}
