package com.paypal.hybris.backoffice.widgets.order.capture;

import com.google.common.base.Predicate;
import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.backoffice.util.PayPalBackofficeUtil;
import com.paypal.hybris.core.exception.PayPalCaptureAdapterException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.service.PayPalManualMultiCaptureService;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.methods.CardPaymentService;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_ZERO_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_INVALID_FORMAT_AMOUNT;


public class PayPalCaptureController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalCaptureController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String REVIEW_DECISION = "_ReviewDecision";

    private OrderModel order;

    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;
    @Wire
    private Combobox transactions;

    @WireVariable
    private transient PaymentService paymentService;

    @WireVariable
    private transient ModelService modelService;

    @WireVariable
    private BusinessProcessService businessProcessService;

    @WireVariable
    private CardPaymentService cardPaymentService;

    @WireVariable
    private I18NService i18nService;

    @WireVariable
    private PayPalManualMultiCaptureService payPalManualMultiCaptureService;

    @SocketEvent(socketId = IN_SOCKET)
    public void initCaptureRequestForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager()
            .setTitle(this.getWidgetInstanceManager().getLabel(PARTIAL_CAPTURE_TITLE)
                + " " + this.getOrder().getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        configureTransactionsCombobox(inputOrder);
    }

    private void configureTransactionsCombobox(final OrderModel inputOrder) {
        List<PaymentTransactionEntryModel> trans = getMultiCaptureableTransactions(inputOrder);
        LOG.info("MultiCaptureable transactions: " + trans);

        for (PaymentTransactionEntryModel v : trans) {
            if(!isSaveOrderFlowActive() || isCaptureAvailable(v)) {
                Comboitem ci = new Comboitem();
                ci.setValue(v);
                ci.setLabel(v.getRequestId());

                transactions.appendChild(ci);
            }
        }

        if (!transactions.getItems().isEmpty()) {
            transactions.setSelectedItem(transactions.getItems().stream().findFirst().get());
            amount.setValue(getAmount(transactions.getSelectedItem().getValue()));
        }
    }

    @ViewEvent(componentID = "transactions", eventName = "onSelect")
    public void selectTransactionEntry() {
        Comboitem ci = transactions.getSelectedItem();
        amount.setValue(getAmount(ci.getValue()));
    }

    @ViewEvent(componentID = "multiplecapturerequest", eventName = "onClick")
    public void confirm() {
        PaymentTransactionEntryModel selectedEntry = new PaymentTransactionEntryModel();
        if (transactions.getSelectedIndex() != -1) {
            selectedEntry = transactions.getSelectedItem().getValue();
        }
        validateAmount();
        processCapture(selectedEntry);
    }

    private void processCapture(PaymentTransactionEntryModel selectedEntry) {
        final BigDecimal processedAmount = new BigDecimal(this.amount.getValue());

        try {
            final PaymentTransactionEntryModel entryModel = doCapture(processedAmount, selectedEntry);

            if (TransactionStatus.ACCEPTED.toString().equals(entryModel.getTransactionStatus())) {
                showMessageBox(getLabel(PARTIAL_CAPTURE_SUCCESS),
                        getLabel(PARTIAL_CAPTURE_TITLE)  + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.INFORMATION);
                sendOutput(OUT_MODIFIED_ITEM, order);
            } else {
                showMessageBox(getLabel(PARTIAL_CAPTURE_ERROR),
                            getLabel(PARTIAL_CAPTURE_TITLE),
                            Messagebox.OK,
                            Messagebox.ERROR);
            }
        } catch (Exception e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            showMessageBox(e.getMessage() != null ? "Capture failed. Error message: " + e.getMessage() : "Capture failed.",
                    getLabel(PARTIAL_CAPTURE_ERROR),
                    Messagebox.OK,
                    Messagebox.ERROR);
        }
    }

    private PaymentTransactionEntryModel doCapture(final BigDecimal amount,
        PaymentTransactionEntryModel selectedEntry) {
        PaymentTransactionEntryModel entry = new PaymentTransactionEntryModel();

        if (order != null) {
            modelService.refresh(getOrder());
            PaymentTransactionType transactionType =
                amount.doubleValue() >= PayPalBackofficeUtil.calculateAuthorizedAmount(
                    getOrder().getPaymentTransactions()).doubleValue() ?
                    PaymentTransactionType.CAPTURE :
                    PaymentTransactionType.PARTIAL_CAPTURE;
            try {
                entry = payPalManualMultiCaptureService.doMultiCapture(amount, selectedEntry, transactionType);
                if(isSaveOrderFlowActive()){
                    entry.getPaymentTransaction().setPlannedAmount(entry.getAmount());
                    modelService.save(entry.getPaymentTransaction());
                }
                if (isOrderFullyCaptured()) {
                    executeManualPaymentCaptureOperation(getOrder());
                }
            } catch (AdapterException e) {
                String message = "Exception, message : " + e.getMessage();
                LOG.error(message, e);
                entry = handlePayPalSDKError(selectedEntry.getPaymentTransaction(), e, amount);
            }
        }
        return entry;
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private void validateAmount() {
        String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(PARTIAL_CAPTURE_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel(PARTIAL_CAPTURE_ZERO_AMOUNT));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(PARTIAL_CAPTURE_INVALID_FORMAT_AMOUNT));
        }
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

    private List<PaymentTransactionEntryModel> getMultiCaptureableTransactions(final OrderModel order) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(paymentEntry -> (PaymentTransactionType.AUTHORIZATION.equals(paymentEntry.getType())
                && TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus())))
            .collect(Collectors.toList());
    }

    private boolean isCaptureAvailable(PaymentTransactionEntryModel authorizationEntry){
       return getOrder().getPaymentTransactions().stream().flatMap(transaction->transaction.getEntries().stream())
           .filter(getCaptureEntriesPredicate())
           .noneMatch(entry -> authorizationEntry.getRequestId().equals(entry.getAuthorizationId()));
    }

    private Predicate<PaymentTransactionEntryModel> getCaptureEntriesPredicate()
    {
        modelService.refresh(getOrder());
        return entry ->
                 (PaymentTransactionType.PARTIAL_CAPTURE.equals(entry.getType()) || PaymentTransactionType.CAPTURE
                        .equals(entry.getType()))
                        && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());
    }

    private boolean isOrderFullyCaptured()
    {
        modelService.refresh(getOrder());
        BigDecimal result = getOrder().getPaymentTransactions().stream()
            .flatMap(transaction -> transaction.getEntries().stream())
            .filter(getCaptureEntriesPredicate()).map(PaymentTransactionEntryModel::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);

        return result.compareTo(isSaveOrderFlowActive() ? calculateTransactionsPlanedAmount()
            : PayPalBackofficeUtil.calculateAuthorizedAmount(
                getOrder().getPaymentTransactions())) >= 0;
    }


    protected void executeManualPaymentCaptureOperation(OrderModel order) {
        order.getOrderProcess().stream()
            .filter(process -> process.getCode().startsWith(order.getStore().getSubmitOrderProcessCode()))
            .forEach(filteredProcess -> businessProcessService
                .triggerEvent(filteredProcess.getOrder().getCode() + REVIEW_DECISION));
        LOG.info(String.format("Payment Capture Manual Release completed. %s triggered.", "ReviewDecision"));

        order.setStatus(OrderStatus.PAYMENT_CAPTURED);
        order.setPaymentStatusType(PaymentStatusType.COMPLETED);
        modelService.save(order);
    }

    protected PaymentTransactionEntryModel handlePayPalSDKError(PaymentTransactionModel transactionToCapture,
                                                              AdapterException e, BigDecimal amount) {
        PaymentTransactionType transactionType = PaymentTransactionType.CAPTURE;
        String newEntryCode = getNewPaymentTransactionEntryCode(transactionToCapture, transactionType);
        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setType, transactionType)
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transactionToCapture)
            .with(PaymentTransactionEntryModel::setAmount, amount)
            .with(PaymentTransactionEntryModel::setCurrency, transactionToCapture.getCurrency())
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.ERROR.toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails, TransactionStatusDetails.INVALID_REQUEST.name())
            .with(PaymentTransactionEntryModel::setFailureReason, e.getMessage())
            .build();

        if (e instanceof PayPalCaptureAdapterException exception) {
            entry.setRequest(PayPalCommandsUtil.getValueAsString(exception.getRequest()));
            entry.setResponse(exception.getMessage());
            entry.setFailureReason(exception.getParentException().getMessage());
            entry.setDebugId(exception.getDebugId());
            entry.setCreateTime(exception.getCreateDate());
        }
        modelService.attach(entry);
        PayPalCommandsUtil.saveTransactionEntryToTransaction(modelService, transactionToCapture, entry);
        return entry;
    }

    public String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
        PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                + 1);
    }

    private BigDecimal getAmountCapturedForTransaction(final PaymentTransactionEntryModel selectedAuthorizationTransaction)
    {
        return getOrder().getPaymentTransactions().stream()
                .flatMap(transaction -> transaction.getEntries().stream())
                .filter(getCaptureEntriesPredicate())
                .filter(getAuthorizationToCapturePredicate(selectedAuthorizationTransaction))
                .map(PaymentTransactionEntryModel::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    protected String getAmount(PaymentTransactionEntryModel selectedAuthorizationTransaction) {
        BigDecimal authorizationTransactionAmount = selectedAuthorizationTransaction.getAmount();
        BigDecimal amountCapturedForTransaction = getAmountCapturedForTransaction(selectedAuthorizationTransaction);
        BigDecimal amountAvailableForCapture = authorizationTransactionAmount.subtract(amountCapturedForTransaction);

        if (amountAvailableForCapture.compareTo(BigDecimal.ZERO) > 0){
            return formatAmount(amountAvailableForCapture);
        }else {
            return formatAmount(BigDecimal.ZERO);
        }
    }

    protected void showMessageBox(String message, String title, int buttons, String icon) {
        Messagebox.show(message, title, buttons, icon);
    }

    private boolean isSaveOrderFlowActive(){
        final PaymentInfoModel paymentInfo = getOrder().getPaymentInfo();
        return paymentInfo instanceof PayPalCreditCardPaymentInfoModel && ((PayPalCreditCardPaymentInfoModel) paymentInfo).isSaveOrderFlowActive();
    }

    private Predicate<PaymentTransactionEntryModel> getAuthorizationToCapturePredicate(final PaymentTransactionEntryModel selectedAuthorizationTransaction){
        return isSaveOrderFlowActive()
            ? entry -> selectedAuthorizationTransaction.getRequestId().equals(entry.getAuthorizationId())
            : entry -> selectedAuthorizationTransaction.getSubscriptionID().equals(entry.getSubscriptionID());
    }

    private BigDecimal calculateTransactionsPlanedAmount(){
        modelService.refresh(getOrder());
        return getOrder().getPaymentTransactions().stream().map(PaymentTransactionModel::getPlannedAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

}
