package com.paypal.hybris.backoffice.widgets.order.authorize;

import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.backoffice.util.PayPalBackofficeUtil;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zk.ui.select.annotation.WireVariable;
import org.zkoss.zul.Label;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_ORDER_AMOUNT_EXCEEDED;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.MANUAL_AUTHORIZATION_ZERO_AMOUNT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.NUMBER_OF_AVAILABLE_AUTHORIZATION_FOR_SAVED_ORDER;


public class PayPalAuthorizationController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(PayPalAuthorizationController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String CONFIRM_TITLE = "paypalbackoffice.authorizeorder.confirm.title";
    private static final String NUMBER_OF_AUTHORIZATION_MESSAGE = "You have %d more authorization attempt%s left.";

    private OrderModel order;
    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;
    @Wire
    private Label saveOrderMessage;
    @WireVariable
    private ModelService modelService;
    @WireVariable
    private PayPalManualAuthorizationService payPalManualAuthorizationService;

    @SocketEvent(socketId = IN_SOCKET)
    public void initCreateAuthorizeRequestForm(OrderModel inputOrder) {
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel)inputOrder.getPaymentInfo();
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager().setTitle(
            this.getWidgetInstanceManager().getLabel(CONFIRM_TITLE) + " " + this
                .getOrder()
                .getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        String totalAmount = paymentInfo.isSaveOrderFlowActive() ? formatAmount(getAmountAvailableForAuthorization())
            : this.getOrder().getTotalPrice().toString();
        this.amount.setValue(totalAmount);
        if(paymentInfo.isSaveOrderFlowActive()){
            this.saveOrderMessage.setValue(generateMessageForSuccessfulAuthorization());
        }

    }

    @ViewEvent(componentID = "authorizeorderrequest", eventName = "onClick")
    public void confirm() {
        validateAmount();
        final BigDecimal confirmedAmount = new BigDecimal(this.amount.getValue());
        processAuthorization(confirmedAmount);
        sendOutput(OUT_MODIFIED_ITEM, order);
    }

    private void processAuthorization(BigDecimal amount) {
        try {
            final PaymentTransactionEntryModel paymentTransactionEntryModel = payPalManualAuthorizationService
                .doAuthorization(getOrder(), amount);

            if (TransactionStatus.ACCEPTED.toString().equals(paymentTransactionEntryModel.getTransactionStatus())) {
                showMessageBox(getLabel(MANUAL_AUTHORIZATION_SUCCESS),
                        getLabel(MANUAL_AUTHORIZATION_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.INFORMATION);
            } else {
                showMessageBox(getLabel(MANUAL_AUTHORIZATION_ERROR) + " : " + paymentTransactionEntryModel.getTransactionStatusDetails(),
                        getLabel(MANUAL_AUTHORIZATION_TITLE) + " " + this.getOrder().getCode(),
                        Messagebox.OK,
                        Messagebox.ERROR);
            }
        } catch (final AdapterException e) {
            order.setPaymentStatusType(PaymentStatusType.FAILED);
            modelService.save(order);
            final String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            showMessageBox("Error message: " + e.getMessage(),
                    getLabel(MANUAL_AUTHORIZATION_TITLE) + " " + this.getOrder().getCode(),
                    Messagebox.OK,
                    Messagebox.ERROR);
        }
    }

    private void validateAmount() {
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel)getOrder().getPaymentInfo();
        final String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_EMPTY_AMOUNT));
        }
        try {
            if (BigDecimal.ZERO.compareTo(new BigDecimal(value)) == 0) {
                throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_ZERO_AMOUNT));
            }
            if (paymentInfo.isSaveOrderFlowActive() && getAmountAvailableForAuthorization().compareTo(new BigDecimal(value)) < 0) {
                throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_ORDER_AMOUNT_EXCEEDED));
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel(MANUAL_AUTHORIZATION_INVALID_FORMAT_AMOUNT));
        }
    }

    private BigDecimal getAmountAvailableForAuthorization(){
        modelService.refresh(getOrder());
        BigDecimal capturedAmount = PayPalBackofficeUtil.calculateCapturedAmount(getOrder().getPaymentTransactions());
        BigDecimal notCapturedAuthorizationAmount = PayPalBackofficeUtil.calculateTransactionsPlanedAmount(
            getOrder().getPaymentTransactions()).subtract(capturedAmount);
        return BigDecimal.valueOf(getOrder().getTotalPrice())
            .subtract(capturedAmount.add(notCapturedAuthorizationAmount));
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private long getNumberOfSuccessfulAuthorization(){
        modelService.refresh(getOrder());
        return getOrder().getPaymentTransactions().stream().flatMap(tr->tr.getEntries().stream())
                .filter(entry -> (PaymentTransactionType.AUTHORIZATION
                    .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()))
                .count();
    }

    private String generateMessageForSuccessfulAuthorization() {
        long numberOfAvailableAuthorization =
            NUMBER_OF_AVAILABLE_AUTHORIZATION_FOR_SAVED_ORDER - getNumberOfSuccessfulAuthorization();
        return String.format(NUMBER_OF_AUTHORIZATION_MESSAGE,
            numberOfAvailableAuthorization,
            numberOfAvailableAuthorization > 1 ? "s" : StringUtils.EMPTY);
    }

    protected void showMessageBox(String message, String title, int buttons, String icon) {
        Messagebox.show(message, title, buttons, icon);
    }


    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }

}
