package com.paypal.hybris.backoffice.action.order.authorize;

import com.hybris.cockpitng.actions.ActionContext;
import com.hybris.cockpitng.actions.ActionResult;
import com.hybris.cockpitng.actions.CockpitAction;
import com.hybris.cockpitng.engine.impl.AbstractComponentWidgetAdapterAware;
import com.paypal.hybris.backoffice.util.PayPalBackofficeUtil;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import java.math.BigDecimal;
import java.util.Map;

import de.hybris.platform.servicelayer.session.SessionExecutionBody;
import de.hybris.platform.servicelayer.session.SessionService;

import javax.annotation.Resource;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.NUMBER_OF_AVAILABLE_AUTHORIZATION_FOR_SAVED_ORDER;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;


public class PayPalAuthorizeOrderAction extends AbstractComponentWidgetAdapterAware implements
    CockpitAction<OrderModel, OrderModel> {

    private static final String SOCKET_OUT_CONTEXT = "paypalAuthorizeOrderContext";
    private static final String CURRENT_SITE = "currentSite";

    @Resource(name = "payPalConfigurationService")
    private PayPalConfigurationService payPalConfigurationService;

    @Resource(name = "sessionService")
    private SessionService sessionService;


    @Override
    public ActionResult<OrderModel> perform(ActionContext<OrderModel> actionContext) {
        OrderModel order = actionContext.getData();
        sendOutput(SOCKET_OUT_CONTEXT, order);
        return new ActionResult<>(ActionResult.SUCCESS);
    }

    @Override
    public boolean canPerform(ActionContext<OrderModel> ctx) {
        final OrderModel order = ctx.getData();
        if(order == null){
            return false;
        }
        return sessionService.executeInLocalViewWithParams(Map.of(CURRENT_SITE, order.getSite()), new SessionExecutionBody() {
            @Override
            public Object execute() {
                boolean result = false;
                if (payPalConfigurationService.getPayPalIntent().equalsIgnoreCase(PAYPAL_INTENT_AUTHORIZE)) {
                    PaymentInfoModel paymentInfo = order.getPaymentInfo();
                    if (!(paymentInfo instanceof PayPalCreditCardPaymentInfoModel)) {
                        result = false;
                    } else {
                        PayPalCreditCardPaymentInfoModel payPalPaymentInfo = (PayPalCreditCardPaymentInfoModel) paymentInfo;
                        result = payPalPaymentInfo.isSaveOrderFlowActive() && canPerformWhenSaveOrderActive(order);
                    }
                }
                return result;
            }
        });
    }

    private boolean canPerformWhenSaveOrderActive(OrderModel orderModel){
        return !OrderStatus.CANCELLED.equals(orderModel.getStatus()) && isAuthorizationAvailableForOrder(orderModel);
    }

    private boolean isAuthorizationAvailableForOrder(OrderModel orderModel) {
        BigDecimal capturedAmount = PayPalBackofficeUtil.calculateCapturedAmount(orderModel.getPaymentTransactions());
        BigDecimal notCapturedAuthorizationAmount = PayPalBackofficeUtil.calculateTransactionsPlanedAmount(
                orderModel.getPaymentTransactions()).subtract(capturedAmount);
        BigDecimal orderAmount = BigDecimal.valueOf(orderModel.getTotalPrice());
        return orderModel.getPaymentTransactions().isEmpty() ||
                orderModel.getPaymentTransactions().stream().flatMap(tr -> tr.getEntries().stream())
                        .filter(entry -> (PaymentTransactionType.AUTHORIZATION
                                .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()))
                        .count() < NUMBER_OF_AVAILABLE_AUTHORIZATION_FOR_SAVED_ORDER
                        && orderAmount.compareTo(capturedAmount.add(notCapturedAuthorizationAmount)) > 0;
    }

}
