package com.paypal.hybris.b2bfacade.impl;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.impl.DefaultPayPalCustomerAccountService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentInfoService;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCreditCardFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.b2bacceleratorservices.customer.impl.DefaultB2BCustomerAccountService;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.core.PK;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.orderscheduling.model.CartToOrderCronJobModel;
import de.hybris.platform.servicelayer.exceptions.ModelNotFoundException;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalB2BCreditCardFacadeTest {

    private static final String PM_ID = "8359490977792";
    private static final String SUBSCRIPTION_ID = "4321";
    private static final String CODE = "CODE";
    private static final PK PK = de.hybris.platform.core.PK.BIG_PK;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private CartToOrderCronJobModel cartToOrderCronJobModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel;
    @Mock
    private UserService userService;
    @Mock
    private UserFacade userFacade;
    @Mock
    private DefaultB2BCustomerAccountService b2bCustomerAccountService;
    @Mock
    private DefaultPayPalCustomerAccountService customerAccountService;
    @Mock
    private DefaultPayPalPaymentInfoService payPalPaymentInfoService;
    @Mock
    private ModelService modelService;

    @InjectMocks
    @Spy
    private DefaultPayPalB2BCreditCardFacade unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, PM_ID)).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalPaymentInfoService.getCreditCardBySubscriptionId(SUBSCRIPTION_ID)).thenReturn(Optional.of(payPalCreditCardPaymentInfoModel));
        when(payPalCreditCardPaymentInfoModel.getUser()).thenReturn(customerModel);
        when(payPalCreditCardPaymentInfoModel.getPk()).thenReturn(PK);
        doReturn(List.of(cartToOrderCronJobModel)).when(b2bCustomerAccountService).getCartToOrderCronJobsForUser(customerModel);
        when(cartToOrderCronJobModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID);
        when(customerModel.getDefaultPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getCode()).thenReturn(CODE);

        when(payPalPaymentInfoService.deletePaypalPaymentToken(SUBSCRIPTION_ID)).thenReturn(Boolean.TRUE);

    }

    @Test
    public void defPayPalCCFacadeIsSuper() {
        Assert.assertTrue(DefaultPayPalCreditCardFacade.class.isAssignableFrom(unit.getClass()));
    }

    @Test
    public void shouldDeleteForPMId() {
        unit.deleteCreditCardFromPayPal(PM_ID);
        verify(payPalPaymentInfoService, times(1)).deletePaypalPaymentToken(SUBSCRIPTION_ID);
        verify(userFacade, times(1)).removeCCPaymentInfo(PM_ID);
        verify(modelService, times(1)).save(customerModel);
    }

    @Test
    public void shouldCancelReplenishmentForPMId() {
        unit.deleteCreditCardFromPayPal(PM_ID);
        verify(cartToOrderCronJobModel).setActive(false);
        verify(modelService).save(cartToOrderCronJobModel);
    }

    @Test
    public void shouldDeletePMWithWebhook() {
        unit.deleteCreditCardFromWebhookEvent(SUBSCRIPTION_ID);
        verify(customerAccountService).deleteCCPaymentInfo(customerModel, payPalCreditCardPaymentInfoModel);
        verify(modelService, times(1)).save(customerModel);
    }

    @Test
    public void shouldCancelReplenishmentFromWebHook() {
        unit.deleteCreditCardFromWebhookEvent(SUBSCRIPTION_ID);
        verify(cartToOrderCronJobModel).setActive(false);
        verify(modelService).save(cartToOrderCronJobModel);
    }

    @Test
    public void shouldDeleteForPMIdNoReplenishments() {
        doReturn(Collections.emptyList()).when(b2bCustomerAccountService).getCartToOrderCronJobsForUser(customerModel);
        unit.deleteCreditCardFromPayPal(PM_ID);
        verify(payPalPaymentInfoService, times(1)).deletePaypalPaymentToken(SUBSCRIPTION_ID);
        verify(userFacade, times(1)).removeCCPaymentInfo(PM_ID);
        verify(modelService, times(1)).save(customerModel);
        verify(cartToOrderCronJobModel, never()).setActive(false);
        verify(modelService, never()).save(cartToOrderCronJobModel);
    }

    @Test
    public void shouldDeletePMWithWebhookNoReplenishments() {
        doReturn(Collections.emptyList()).when(b2bCustomerAccountService).getCartToOrderCronJobsForUser(customerModel);
        unit.deleteCreditCardFromWebhookEvent(SUBSCRIPTION_ID);
        verify(customerAccountService).deleteCCPaymentInfo(customerModel, payPalCreditCardPaymentInfoModel);
        verify(modelService, times(1)).save(customerModel);
        verify(cartToOrderCronJobModel, never()).setActive(false);
        verify(modelService, never()).save(cartToOrderCronJobModel);
    }

    @Test(expected = ModelNotFoundException.class)
    public void shouldDoNothingWhenPMIdInvalid() {
        unit.deleteCreditCardFromPayPal("0000");
        verify(payPalPaymentInfoService, never()).deletePaypalPaymentToken(SUBSCRIPTION_ID);
        verify(userFacade, never()).removeCCPaymentInfo(PM_ID);
        verify(modelService, never()).save(customerModel);
        verify(cartToOrderCronJobModel, never()).setActive(false);
        verify(modelService, never()).save(cartToOrderCronJobModel);
    }

    @Test
    public void shouldDoNothingWhenWebhookSubscriptionIdInvalid() {
        unit.deleteCreditCardFromWebhookEvent("000000");
        verify(customerAccountService, never()).deleteCCPaymentInfo(customerModel, payPalCreditCardPaymentInfoModel);
        verify(modelService, never()).save(customerModel);
        verify(cartToOrderCronJobModel, never()).setActive(false);
        verify(modelService, never()).save(cartToOrderCronJobModel);
    }

}
