package com.paypal.hybris.b2bfacade.impl;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import com.paypal.hybris.data.PayPalPlaceOrderFormData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorfacades.order.AcceleratorCheckoutFacade;
import de.hybris.platform.b2b.model.B2BCommentModel;
import de.hybris.platform.b2b.services.B2BCommentService;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BPaymentTypeData;
import de.hybris.platform.b2bacceleratorservices.order.B2BCommerceCartService;
import de.hybris.platform.commercefacades.order.CartFacade;
import de.hybris.platform.commercefacades.order.data.AbstractOrderData;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.DeliveryModeData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.order.CommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.commerceservices.service.data.CommerceOrderResult;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.HybrisEnumValue;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.enumeration.EnumerationService;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.ArrayList;
import java.util.Collections;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;


@UnitTest
public class DefaultPayPalB2BCheckoutFacadeTest {

    private static final String CHECKOUT_PAYMENT_TYPE = "CheckoutPaymentType";
    private static final String SECURITY_CODE = "12345";
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String CARD = "CARD";

    @InjectMocks
    @Spy
    private DefaultPayPalB2BCheckoutFacade unit;

    @Mock
    private AcceleratorCheckoutFacade acceleratorCheckoutFacade;
    @Mock
    private CartFacade cartFacade;
    @Mock
    private CartService cartService;
    @Mock
    private CartModel cartModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfo;
    @Mock
    private CreditCardPaymentInfoModel defaultPaymentInfo;
    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;
    @Mock
    private CommerceCheckoutService commerceCheckoutService;
    @Mock
    private PaymentTransactionEntryModel paymentTransactionEntryModel;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private CartData cartData;
    @Mock
    private Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter;
    @Mock
    private EnumerationService enumerationService;
    @Mock
    private Converter<HybrisEnumValue, B2BPaymentTypeData> b2bPaymentTypeDataConverter;
    @Mock
    private PlaceOrderData placeOrderData;
    @Mock
    private ModelService modelService;
    @Mock
    private B2BCommentService<AbstractOrderModel> b2bCommentService; //to avoid NPE
    @Mock
    private Converter<OrderModel, OrderData> orderConverter;
    @Mock
    private B2BCommerceCartService commerceCartService;
    @Spy
    private ArrayList<HybrisEnumValue> hybrisEnumValues;
    @Mock
    private OrderData orderData;
    @Mock
    private OrderModel orderModel;
    @Mock
    private CommerceOrderResult orderResult;
    @Mock
    private B2BCommentModel b2bCommentModel;
    private PayPalPlaceOrderFormData placeOrderFormData;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        placeOrderFormData = new PayPalPlaceOrderFormData();
        placeOrderFormData.setSecurityCode(SECURITY_CODE);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfo);
        when(cartModel.getUser()).thenReturn(customerModel);
    }

    @Test
    public void shouldCalculateCartWhenPaymentTypeWasChanged() {
        when(enumerationService.getEnumerationValues(CHECKOUT_PAYMENT_TYPE)).thenReturn(hybrisEnumValues);
        when(hybrisEnumValues.contains(any(HybrisEnumValue.class))).thenReturn(true);

        unit.setCardPaymentType();
        verify(commerceCartService).calculateCartForPaymentTypeChange(cartModel);
    }

    @Test
    public void shouldReturnFalseWhenCartIsNull() {
        when(cartFacade.getSessionCart()).thenReturn(null);
        when(cartService.getSessionCart()).thenReturn(null);
        assertFalse(unit.authorizePayment(placeOrderFormData));
    }

    @Test
    public void shouldAuthorizePaymentWhenNonPayPalCreditCardPaymentInfoModel() {
        when(cartModel.getPaymentInfo()).thenReturn(defaultPaymentInfo);
        doReturn(true).when(unit).authorizePayment(SECURITY_CODE);

        assertTrue(unit.authorizePayment(placeOrderFormData));
        verify(unit).authorizePayment(SECURITY_CODE);
    }

    @Test
    public void shouldAuthorizePaymentWhenSaveOrderFlowActive() {
        when(paymentInfo.isSaveOrderFlowActive()).thenReturn(true);
        when(acceleratorCheckoutFacade.authorizePayment(SECURITY_CODE)).thenReturn(true);

        assertTrue(unit.authorizePayment(placeOrderFormData));
        verify(acceleratorCheckoutFacade).authorizePayment(SECURITY_CODE);
    }

    @Test
    public void shouldAuthorizePaymentWhenSaveOrderFlowNotActive() {
        when(paymentInfo.isSaveOrderFlowActive()).thenReturn(false);
        when(paymentInfo.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(commerceCheckoutService.authorizePayment(any(PayPalCommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        assertTrue(unit.authorizePayment(placeOrderFormData));
        verify(commerceCheckoutService).authorizePayment(any(PayPalCommerceCheckoutParameter.class));
    }

    @Test
    public void shouldPlaceOrderWhenNegotiationPresent() throws InvalidCartException {
        preparePlaceOrderTest();

        when(placeOrderData.getNegotiateQuote()).thenReturn(true);
        when(placeOrderData.getQuoteRequestDescription()).thenReturn("quote description");
        when(orderResult.getOrder()).thenReturn(orderModel);
        when(commerceCheckoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(orderConverter.convert(orderModel)).thenReturn(orderData);

        AbstractOrderData placedOrder = unit.placeOrder(placeOrderData);

        assertEquals(orderData, placedOrder);
        verify(modelService).save(cartModel);
        verify(orderConverter).convert(orderModel);
        verify(commerceCheckoutService).placeOrder(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldPlaceOrderWhenSaveOrderFlowNotActive() throws InvalidCartException {
        preparePlaceOrderTest();

        when(paymentInfo.isSaveOrderFlowActive()).thenReturn(false);
        when(orderResult.getOrder()).thenReturn(orderModel);
        when(commerceCheckoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(orderConverter.convert(orderModel)).thenReturn(orderData);

        AbstractOrderData placedOrder = unit.placeOrder(placeOrderData);

        assertEquals(orderData, placedOrder);
        verify(orderConverter).convert(orderModel);
        verify(commerceCheckoutService).placeOrder(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldReturnFalseWhenSubscriptionIdIsEmpty() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfo);
        when(paymentInfo.getSubscriptionId()).thenReturn("");
        when(commerceCheckoutService.authorizePayment(any(PayPalCommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        assertFalse(unit.authorizeB2BPayment(placeOrderFormData));
    }

    @Test
    public void shouldReturnTrueWhenTransactionReturnedReviewStatus() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfo);
        when(paymentInfo.isSaveOrderFlowActive()).thenReturn(false);
        when(paymentInfo.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(commerceCheckoutService.authorizePayment(any(PayPalCommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REVIEW.name());

        assertTrue(unit.authorizeB2BPayment(placeOrderFormData));
    }

    private void preparePlaceOrderTest() {
        B2BPaymentTypeData paymentTypeData = new B2BPaymentTypeData();
        paymentTypeData.setCode(CARD);

        when(paymentInfo.isSaveOrderFlowActive()).thenReturn(true);
        when(cartData.isCalculated()).thenReturn(true);
        when(cartData.getDeliveryAddress()).thenReturn(new AddressData());
        when(cartData.getDeliveryMode()).thenReturn(new DeliveryModeData());
        when(cartData.getQuoteAllowed()).thenReturn(true);
        when(cartData.getPaymentType()).thenReturn(paymentTypeData);
        when(cartData.getPaymentInfo()).thenReturn(new CCPaymentInfoData());
        when(creditCardPaymentInfoConverter.convert(any(CreditCardPaymentInfoModel.class))).thenReturn(new CCPaymentInfoData());
        when(b2bPaymentTypeDataConverter.convert(any())).thenReturn(paymentTypeData);
        when(modelService.create(B2BCommentModel.class)).thenReturn(b2bCommentModel);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);

        doReturn(Collections.singletonList(paymentTypeData)).when(unit).getPaymentTypes();
    }

}
