/*

 */
package com.paypal.hybris.b2bfacade.impl;

import com.paypal.hybris.b2bfacade.PayPalB2BCheckoutFacade;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import com.paypal.hybris.data.PayPalPlaceOrderFormData;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.exception.EntityValidationException;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BCommentData;
import de.hybris.platform.b2bacceleratorfacades.order.impl.DefaultB2BAcceleratorCheckoutFacade;
import de.hybris.platform.commercefacades.order.data.AbstractOrderData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Optional;

import static de.hybris.platform.util.localization.Localization.getLocalizedString;

/**
 * This class is a default implementation of the PayPalB2BCheckoutFacade interface
 */
public class DefaultPayPalB2BCheckoutFacade extends DefaultB2BAcceleratorCheckoutFacade implements
        PayPalB2BCheckoutFacade {

    private static final String CART_CHECKOUT_NO_QUOTE_DESCRIPTION = "cart.no.quote.description";

    @Override
    public void setCardPaymentType() {
        setPaymentTypeForCart("CARD", getCart());
    }

    @Override
    public boolean authorizePayment(final PayPalPlaceOrderFormData formData) {
        final CartModel cart = getCart();
        if (cart == null) {
            return false;
        }
        if (!isPayPalCreditCardPaymentInfoInstance(cart)) {
            return super.authorizePayment(formData.getSecurityCode());
        }
        PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfo = (PayPalCreditCardPaymentInfoModel) cart.getPaymentInfo();
        if (payPalCreditCardPaymentInfo.isSaveOrderFlowActive()) {
            return getAcceleratorCheckoutFacade().authorizePayment(formData.getSecurityCode());
        }
        return this.authorizeB2BPayment(formData);
    }

    private static boolean isPayPalCreditCardPaymentInfoInstance(CartModel cart) {
        return cart.getPaymentInfo() instanceof PayPalCreditCardPaymentInfoModel;
    }

    @Override
    public <T extends AbstractOrderData> T placeOrder(final PlaceOrderData placeOrderData) throws InvalidCartException {
        final PaymentInfoModel paymentInfo = getCart().getPaymentInfo();
        if (paymentInfo instanceof PayPalCreditCardPaymentInfoModel
                && ((PayPalCreditCardPaymentInfoModel) paymentInfo).isSaveOrderFlowActive()) {
            return placeOrderWithNegotiation(placeOrderData);
        } else {
            return super.placeOrder(placeOrderData);
        }

    }

    private <T extends AbstractOrderData> T placeOrderWithNegotiation(final PlaceOrderData placeOrderData) throws InvalidCartException {
        if (isValidCheckoutCart(placeOrderData)) {
            if (BooleanUtils.isTrue(placeOrderData.getNegotiateQuote())) {
                if (StringUtils.isBlank(placeOrderData.getQuoteRequestDescription())) {
                    throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_NO_QUOTE_DESCRIPTION));
                } else {
                    final B2BCommentData b2BComment = new B2BCommentData();
                    b2BComment.setComment(placeOrderData.getQuoteRequestDescription());
                    final CartData cartData = new CartData();
                    cartData.setB2BComment(b2BComment);
                    updateCheckoutCart(cartData);
                }
            }
            return (T) super.placeOrder();
        } else {
            return null;
        }

    }

    public boolean authorizeB2BPayment(final PayPalPlaceOrderFormData formData) {
        final CartModel cartModel = getCart();
        final CreditCardPaymentInfoModel creditCardPaymentInfoModel = Optional.ofNullable(cartModel)
                .map(CartModel::getPaymentInfo)
                .map(CreditCardPaymentInfoModel.class::cast)
                .orElse(null);
        if (checkIfCurrentUserIsTheCartUser() && creditCardPaymentInfoModel != null
                && StringUtils.isNotBlank(creditCardPaymentInfoModel.getSubscriptionId())) {
            final PayPalCommerceCheckoutParameter parameter = createPayPalCommerceCheckoutParameter(cartModel, formData);

            final PaymentTransactionEntryModel paymentTransactionEntryModel;
            paymentTransactionEntryModel = getCommerceCheckoutService().authorizePayment(parameter);

            return paymentTransactionEntryModel != null
                    && (TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())
                    || TransactionStatus.REVIEW.name().equals(paymentTransactionEntryModel.getTransactionStatus()));
        }
        return false;
    }

    private PayPalCommerceCheckoutParameter createPayPalCommerceCheckoutParameter(final CartModel cart, final PayPalPlaceOrderFormData formData) {
        return GenericBuilder.of(PayPalCommerceCheckoutParameter::new)
                .with(PayPalCommerceCheckoutParameter::setEnableHooks, true)
                .with(PayPalCommerceCheckoutParameter::setCart, cart)
                .with(PayPalCommerceCheckoutParameter::setReplenishment, formData.isReplenishment())
                .with(PayPalCommerceCheckoutParameter::setPaymentProvider, getPaymentProvider())
                .with(PayPalCommerceCheckoutParameter::setSecurityCode, formData.getSecurityCode())
                .build();
    }

}
