package com.paypalocc.helpers;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.OrderFacade;
import de.hybris.platform.commercefacades.order.data.OrderHistoriesData;
import de.hybris.platform.commercefacades.order.data.OrderHistoryData;
import de.hybris.platform.commerceservices.search.pagedata.PageableData;
import de.hybris.platform.commerceservices.search.pagedata.PaginationData;
import de.hybris.platform.commerceservices.search.pagedata.SearchPageData;
import de.hybris.platform.commerceservices.search.pagedata.SortData;
import de.hybris.platform.commercewebservicescommons.dto.order.OrderHistoryListWsDTO;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.webservicescommons.mapping.DataMapper;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class OrdersHelperTest {

	private static final int CURRENT_PAGE = 3;
	private static final int PAGE_SIZE = 3;
	private static final String SORT = "sort";
	private static final String STATUSES = "APPROVED,CREATED";
	private static final String FIELDS = "fields";

	private List<OrderHistoryData> orderHistoryDataList;
	private List<SortData> sortDataList;
	private PaginationData paginationData;
	private SearchPageData<OrderHistoryData> result;

	@Mock
	private OrderFacade orderFacade;
	@Mock
	private DataMapper dataMapper;
	@InjectMocks
	@Spy
	private OrdersHelper ordersHelper;

	@Before
	public void setUp() throws Exception {
		MockitoAnnotations.initMocks(this);

		orderHistoryDataList = Collections.singletonList(new OrderHistoryData());
		sortDataList = Collections.singletonList(new SortData());
		paginationData = new PaginationData();
		result = new SearchPageData<>();
		result.setResults(orderHistoryDataList);
		result.setSorts(sortDataList);
		result.setPagination(paginationData);
	}

	@Test
	public void shouldCreatePageableData() {

		PageableData pageableData = ordersHelper.createPageableData(CURRENT_PAGE, PAGE_SIZE, SORT);

		assertThat(pageableData.getCurrentPage()).isEqualTo(CURRENT_PAGE);
		assertThat(pageableData.getPageSize()).isEqualTo(PAGE_SIZE);
		assertThat(pageableData.getSort()).isEqualTo(SORT);
	}

	@Test
	public void shouldReturnSetOfOrderStatusesFromString() {

		Set<OrderStatus> resultStatusSet = ordersHelper.extractOrderStatuses(STATUSES);

		assertThat(resultStatusSet).containsExactlyInAnyOrder(OrderStatus.APPROVED, OrderStatus.CREATED);
	}

	@Test
	public void shouldCreateOrderHistoriesDataFromSearchPageData() {

		OrderHistoriesData orderHistoriesData = ordersHelper.createOrderHistoriesData(result);

		assertOrderHistoriesDataContainProperValues(orderHistoriesData);
	}

	@Test
	public void shouldReturnOrderHistoriesForStatusesWhenStatusesNotNull() {
		when(orderFacade.getPagedOrderHistoryForStatuses(any(), eq(OrderStatus.APPROVED), eq(OrderStatus.CREATED)))
				.thenReturn(result);

		OrderHistoriesData orderHistoriesData
				= ordersHelper.searchOrderHistory(STATUSES, CURRENT_PAGE, PAGE_SIZE, SORT);

		assertOrderHistoriesDataContainProperValues(orderHistoriesData);

		verify(orderFacade, never()).getPagedOrderHistoryForStatuses(paginationData);
	}


	@Test
	public void shouldReturnOrderHistoriesWhenStatusesNull() {
		when(orderFacade.getPagedOrderHistoryForStatuses(any())).thenReturn(result);
		OrderHistoriesData orderHistoriesData
				= ordersHelper.searchOrderHistory(null, CURRENT_PAGE, PAGE_SIZE, SORT);

		assertOrderHistoriesDataContainProperValues(orderHistoriesData);

		verify(orderFacade, never()).getPagedOrderHistoryForStatuses(eq(paginationData), any());
	}

	@Test
	public void shouldMapOrderHistoriesDataToReturnOrderHistoryListWsDTO() {
		OrderHistoriesData orderHistoriesData = new OrderHistoriesData();
		OrderHistoryListWsDTO orderHistoryListWsDTOMap = new OrderHistoryListWsDTO();
		doReturn(orderHistoryListWsDTOMap).when(dataMapper).map(orderHistoriesData, OrderHistoryListWsDTO.class, FIELDS);
		doReturn(orderHistoriesData).when(ordersHelper).searchOrderHistory(STATUSES, CURRENT_PAGE, PAGE_SIZE, SORT);

		OrderHistoryListWsDTO orderHistoryListWsDTO
				= ordersHelper.searchOrderHistory(STATUSES, CURRENT_PAGE, PAGE_SIZE, SORT, FIELDS);

		verify(ordersHelper).searchOrderHistory(STATUSES, CURRENT_PAGE, PAGE_SIZE, SORT);
		verify(dataMapper).map(orderHistoriesData, OrderHistoryListWsDTO.class, FIELDS);
		assertThat(orderHistoryListWsDTO).isEqualTo(orderHistoryListWsDTOMap);
	}

	private void assertOrderHistoriesDataContainProperValues(OrderHistoriesData orderHistoriesData) {
		assertThat(orderHistoriesData.getOrders()).isEqualTo(orderHistoryDataList);
		assertThat(orderHistoriesData.getSorts()).isEqualTo(sortDataList);
		assertThat(orderHistoriesData.getPagination()).isEqualTo(paginationData);
	}

}
