package com.paypalocc.validators;

import com.paypal.enums.PayPalPaymentProvider;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;
import org.springframework.validation.Validator;

import javax.validation.constraints.NotNull;
import java.util.Calendar;


public class PayPalCCPaymentInfoValidator implements Validator {

    private static final String FIELD_REQUIRED_MESSAGE_ID = "field.required";
    private static final String START_MONTH = "startMonth";
    private static final String PAYMENT_START_DATE_INVALID = "payment.startDate.invalid";
    private static final String PAYER_EMAIL = "payerEmail";
    private static final String ACCOUNT_HOLDER_NAME = "accountHolderName";
    private static final String CARD_TYPE = "cardType";
    private static final String CARD_NUMBER = "cardNumber";
    private static final String EXPIRY_MONTH = "expiryMonth";
    private static final String EXPIRY_YEAR = "expiryYear";

    private Validator paymentAddressValidator;

    @Override
    public boolean supports(final Class clazz) {
        return CCPaymentInfoData.class.isAssignableFrom(clazz);
    }

    @Override
    public void validate(@NotNull Object target, @NotNull Errors errors) {
        if (target instanceof CCPaymentInfoData ccPaymentData) {
            if (isDatePresent(ccPaymentData)) {
                final Calendar start = Calendar.getInstance();
                start.set(Calendar.DAY_OF_MONTH, 0);
                start.set(Calendar.MONTH, Integer.parseInt(ccPaymentData.getStartMonth()) - 1);
                start.set(Calendar.YEAR, Integer.parseInt(ccPaymentData.getStartYear()) - 1);

                final Calendar expiration = Calendar.getInstance();
                expiration.set(Calendar.DAY_OF_MONTH, 0);
                expiration.set(Calendar.MONTH, Integer.parseInt(ccPaymentData.getExpiryMonth()) - 1);
                expiration.set(Calendar.YEAR, Integer.parseInt(ccPaymentData.getExpiryYear()) - 1);

                if (start.after(expiration)) {
                    errors.rejectValue(START_MONTH, PAYMENT_START_DATE_INVALID);
                }
            }
            if (ObjectUtils.equals(PayPalPaymentProvider.PAYPAL, ccPaymentData.getPaymentProvider())) {
                ValidationUtils.rejectIfEmptyOrWhitespace(errors, PAYER_EMAIL, FIELD_REQUIRED_MESSAGE_ID);
            } else {
                ValidationUtils.rejectIfEmptyOrWhitespace(errors, ACCOUNT_HOLDER_NAME, FIELD_REQUIRED_MESSAGE_ID);
                paymentAddressValidator.validate(ccPaymentData, errors);
            }
            ValidationUtils.rejectIfEmptyOrWhitespace(errors, CARD_TYPE, FIELD_REQUIRED_MESSAGE_ID);
            ValidationUtils.rejectIfEmptyOrWhitespace(errors, CARD_NUMBER, FIELD_REQUIRED_MESSAGE_ID);
            ValidationUtils.rejectIfEmptyOrWhitespace(errors, EXPIRY_MONTH, FIELD_REQUIRED_MESSAGE_ID);
            ValidationUtils.rejectIfEmptyOrWhitespace(errors, EXPIRY_YEAR, FIELD_REQUIRED_MESSAGE_ID);
        }
    }

    private static boolean isDatePresent(CCPaymentInfoData ccPaymentData) {
        return StringUtils.isNotBlank(ccPaymentData.getStartMonth()) &&
                StringUtils.isNotBlank(ccPaymentData.getStartYear())
                && StringUtils.isNotBlank(ccPaymentData.getExpiryMonth()) &&
                StringUtils.isNotBlank(ccPaymentData.getExpiryYear());
    }

    public void setPaymentAddressValidator(Validator paymentAddressValidator) {
        this.paymentAddressValidator = paymentAddressValidator;
    }

}
