package com.paypalocc.controllers;

import com.google.gson.Gson;
import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.errors.PaypalValidationError;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.data.ws.CreditCardAddressWsDTO;
import com.paypal.hybris.data.ws.PaymentTokenWsDTO;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCustomerFacade;
import com.paypalocc.validators.PaypalExpiredDateDtoValidator;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.dto.error.ErrorListWsDTO;
import de.hybris.platform.webservicescommons.dto.error.ErrorWsDTO;
import de.hybris.platform.webservicescommons.errors.exceptions.WebserviceValidationException;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.client.HttpClientErrorException;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;


@Controller
@RequestMapping(value = "/{baseSiteId}/users/{userId}/cards")
@CacheControl(directive = CacheControlDirective.PRIVATE)
@Tag(name = "Save Cards")
public class SaveCreditCardController extends PayPalBaseController {

    private static final String YEAR_FIELD = "YEAR";
    private static final String MONTH_FIELD = "MONTH";
    private static final String NUMBER_FIELD = "NUMBER";
    private static final String NAME_FIELD = "NAME";
    private static final String EXPIRY_YEAR = "expiryYear";
    private static final String EXPIRY_MONTH = "expiryMonth";
    private static final String CARD_NUMBER = "cardNumber";
    private static final String ACCOUNT_HOLDER_NAME = "accountHolderName";

    @Resource(name = "addressDTOValidator")
    private Validator addressDTOValidator;

    @Resource
    private UserFacade userFacade;

    @Resource
    private PayPalCreditCardFacade payPalCreditCardFacade;
    @Resource
    private PaypalExpiredDateDtoValidator paypalExpiredDateDtoValidator;
    @Resource
    private PayPalPaymentInfoService payPalPaymentInfoService;
    @Resource
    private DefaultPayPalCustomerFacade payPalCustomerFacade;


    @Secured({"ROLE_CLIENT", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_CUSTOMERGROUP"})
    @RequestMapping(value = "/setup-token", method = RequestMethod.POST,
            consumes = {MediaType.APPLICATION_JSON_VALUE}, produces = {MediaType.APPLICATION_JSON_VALUE})
    @ResponseStatus(value = HttpStatus.CREATED)
    @ResponseBody
    @Operation(operationId = "createSetupToken", summary = "Create setup token")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentTokenWsDTO createSetupToken(
            @Parameter(description = "CreditCardAddress object.", required = true)
            @RequestBody CreditCardAddressWsDTO creditCardAddress,
            @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) String fields) {
        validate(creditCardAddress, "expiryDate", paypalExpiredDateDtoValidator);
        CreditCardAddressData creditCardAddressData = getDataMapper().map(creditCardAddress, CreditCardAddressData.class);
        boolean isNewAddressShouldBeCreated = creditCardAddress.getSelectedAddressCode() == null;

        if (isNewAddressShouldBeCreated) {
            validate(creditCardAddress.getBillingAddress(), "address", addressDTOValidator);
            AddressData billingAddress = getDataMapper().map(creditCardAddress.getBillingAddress(), AddressData.class);
            creditCardAddressData.setBillingAddress(billingAddress);

            billingAddress.setVisibleInAddressBook(false);
            billingAddress.setShippingAddress(true);
            userFacade.addAddress(billingAddress);

            creditCardAddressData.setSelectedAddressCode(billingAddress.getId());
        } else {
            AddressData addressForCode = userFacade.getAddressForCode(creditCardAddress.getSelectedAddressCode());
            creditCardAddressData.setBillingAddress(addressForCode);
        }

        boolean isAddressDuplicate = false;

        if (shouldAddBillingAddressToAddressBook(creditCardAddress)) {
            AddressData newShippingAddress = getNewShippingAddress(creditCardAddress);
            if (!payPalCustomerFacade.addShippingAddressToCurrentUser(newShippingAddress)) {
                isAddressDuplicate = true;
            }
        }

        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setCreditCardData(creditCardAddressData);
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);

        CCSetupTokenData ccSetupTokenData = payPalCreditCardFacade.requestSetupToken(setupTokenRequestData);

        PaymentTokenWsDTO paymentTokenWsDTO = getDataMapper().map(ccSetupTokenData, PaymentTokenWsDTO.class);
        paymentTokenWsDTO.setIsAddressDuplicate(isAddressDuplicate);
        return paymentTokenWsDTO;
    }

    @Secured({"ROLE_CLIENT", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_CUSTOMERGROUP"})
    @RequestMapping(value = "/payment-token", method = RequestMethod.POST,
            consumes = {MediaType.APPLICATION_JSON_VALUE}, produces = {MediaType.APPLICATION_JSON_VALUE})
    @ResponseStatus(value = HttpStatus.CREATED)
    @ResponseBody
    @Operation(operationId = "createPaymentToken", summary = "Create payment token")
    @ApiBaseSiteIdAndUserIdParam
    public ResponseEntity<String> createPaymentToken(
            @Parameter(description = "CreditCardAddress object.", required = true)
            @RequestBody PaymentTokenWsDTO paymentTokenWsDTO) throws Exception {
        PayPalCreditCardPaymentInfoModel paymentInfoByCode = payPalPaymentInfoService.getPaymentInfoByPK(
                paymentTokenWsDTO.getPaymentInfoPK())
                .orElseThrow(Exception::new);

        PaymentTokenData paymentTokenData = preparePaypalTokenDataForRequest(paymentInfoByCode);

        payPalCreditCardFacade.requestPaymentToken(paymentTokenData);

        return new ResponseEntity<>(HttpStatus.OK);
    }

    @Secured({"ROLE_CLIENT", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_CUSTOMERGROUP"})
    @DeleteMapping(consumes = {MediaType.APPLICATION_JSON_VALUE}, produces = {MediaType.APPLICATION_JSON_VALUE})
    @ResponseBody
    @Operation(operationId = "deletePaymentMethod", summary = "Delete payment method")
    @ApiBaseSiteIdAndUserIdParam
    public ResponseEntity<String> deletePaymentMethod(@RequestParam String pk) {
        payPalPaymentInfoService.removePaymentInfoByPK(pk);

        return new ResponseEntity<>(HttpStatus.OK);
    }

    private PaymentTokenData preparePaypalTokenDataForRequest(PayPalCreditCardPaymentInfoModel creditCardPaymentInfo) {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setSetupToken(creditCardPaymentInfo.getSubscriptionId());
        paymentTokenData.setAddressId(creditCardPaymentInfo.getBillingAddress().getPk().getLongValueAsString());
        paymentTokenData.setPaymentInfoPK(creditCardPaymentInfo.getPk().getLongValueAsString());
        paymentTokenData.setCustomerId(((CustomerModel) creditCardPaymentInfo.getUser()).getVaultCustomerId());

        return paymentTokenData;
    }

    @ResponseStatus(value = HttpStatus.BAD_REQUEST)
    @ResponseBody
    @ExceptionHandler({HttpClientErrorException.class})
    public ErrorListWsDTO handleClientSideException(HttpClientErrorException ex) {
        ErrorListWsDTO errorListDto = new ErrorListWsDTO();
        List<ErrorWsDTO> errors = new ArrayList<>();
        PaypalValidationError paypalValidationErrors = new Gson().fromJson(ex.getResponseBodyAsString(),
                PaypalValidationError.class);
        paypalValidationErrors.getDetails().forEach(errorDetail -> {
            ErrorWsDTO error = new ErrorWsDTO();
            error.setSubject(getValidationErrorSubject(errorDetail.getField()));
            error.setMessage(errorDetail.getDescription());
            error.setType(errorDetail.getIssue());
            errors.add(error);
        });
        errorListDto.setErrors(errors);
        return errorListDto;
    }

    @ResponseStatus(value = HttpStatus.BAD_REQUEST)
    @ResponseBody
    @ExceptionHandler({WebserviceValidationException.class})
    public ErrorListWsDTO handleValidationException(WebserviceValidationException ex) {
        ErrorListWsDTO errorListDto = new ErrorListWsDTO();
        List<ErrorWsDTO> errors = new ArrayList<>();
        ((BeanPropertyBindingResult) (ex.getValidationObject())).getFieldErrors().forEach(fieldError -> {
            ErrorWsDTO error = new ErrorWsDTO();
            error.setSubject(getValidationErrorSubject(fieldError.getField()));
            errors.add(error);
        });

        errorListDto.setErrors(errors);
        return errorListDto;
    }

    private String getValidationErrorSubject(String field) {
        String fieldInUpperCase = field.toUpperCase();
        if (fieldInUpperCase.contains(YEAR_FIELD)) {
            return EXPIRY_YEAR;
        } else if (fieldInUpperCase.contains(MONTH_FIELD)) {
            return EXPIRY_MONTH;
        } else if (fieldInUpperCase.contains(NUMBER_FIELD)) {
            return CARD_NUMBER;
        } else if (fieldInUpperCase.contains(NAME_FIELD)) {
            return ACCOUNT_HOLDER_NAME;
        }
        return StringUtils.EMPTY;
    }

    private boolean shouldAddBillingAddressToAddressBook(CreditCardAddressWsDTO creditCardAddressWsDTO) {
        return creditCardAddressWsDTO.getBillingAddress() != null
                && creditCardAddressWsDTO.getBillingAddress().getIsAddBillingAddressToAddressBook();
    }

    private AddressData getNewShippingAddress(CreditCardAddressWsDTO creditCardAddress) {
        return getDataMapper().map(creditCardAddress.getBillingAddress(), AddressData.class);
    }

}
