/*

 */
package com.paypalocc.controllers;

import com.paypal.hybris.data.ws.PayPalSetUpPasswordFormWsData;
import com.paypal.hybris.facade.exception.PayPalCustomerHasPayerIdException;
import com.paypal.hybris.facade.exception.PayPalDuplicateUidException;
import com.paypal.hybris.facade.exception.PayPalGuestCustomerHasPayerIdException;
import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import com.paypal.hybris.facade.facades.PayPalGuestCheckoutFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalSetUpPasswordFacade;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commercefacades.user.data.RegisterData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.commerceservices.request.mapping.annotation.RequestMappingOverride;
import de.hybris.platform.commercewebservicescommons.annotation.CaptchaAware;
import de.hybris.platform.commercewebservicescommons.annotation.SecurePortalUnauthenticatedAccess;
import de.hybris.platform.commercewebservicescommons.constants.CommercewebservicescommonsConstants;
import de.hybris.platform.commercewebservicescommons.dto.user.UserSignUpWsDTO;
import de.hybris.platform.commercewebservicescommons.dto.user.UserWsDTO;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.RequestParameterException;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;

import java.nio.charset.StandardCharsets;
import java.util.Locale;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.UriUtils;

import static com.paypalocc.constants.PaypaloccWebConstants.LOCATION;

@Controller
@RequestMapping(value = "/{baseSiteId}/users")
@CacheControl(directive = CacheControlDirective.PRIVATE)
@Tag(name = "PayPal Users")
public class PayPalUsersController extends PayPalBaseController {

    private static final Logger LOG = LoggerFactory.getLogger(PayPalUsersController.class);

    @Resource(name = "userSignUpDTOValidator")
    private Validator userSignUpDTOValidator;

    @Resource(name = "wsCustomerFacade")
    private CustomerFacade customerFacade;

    @Resource(name = "payPalSetUpPasswordFacade")
    private DefaultPayPalSetUpPasswordFacade defaultPayPalSetUpPasswordFacade;

    @Resource(name = "passwordStrengthValidator")
    private Validator passwordStrengthValidator;

    @Resource(name = "defaultUserService")
    private UserService userService;

    @Resource(name = "payPalCustomerFacade")
    private PayPalCustomerFacade payPalCustomerFacade;

    @Resource(name = "HttpRequestUserSignUpDTOPopulator")
    private Populator<HttpServletRequest, UserSignUpWsDTO> httpRequestUserSignUpDTOPopulator;

    @Resource(name = "guestConvertingDTOValidator")
    private Validator guestConvertingDTOValidator;

    @Resource(name = "payPalGuestCheckoutFacade")
    private PayPalGuestCheckoutFacade payPalGuestCheckoutFacade;

    @Secured({"ROLE_CLIENT", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @RequestMapping(method = RequestMethod.POST, consumes = {MediaType.APPLICATION_JSON_VALUE,
            MediaType.APPLICATION_XML_VALUE})
    @RequestMappingOverride
    @ResponseStatus(value = HttpStatus.CREATED)
    @ResponseBody
    @Operation(operationId = "createUser", summary = " Registers a customer", description =
            "Registers a customer. Requires the following "
                    + "parameters: login, password, firstName, lastName, titleCode.")
    @ApiBaseSiteIdParam
    public UserWsDTO createUser(
            @Parameter(description = "User's object.", required = true) @RequestBody final UserSignUpWsDTO user,
            @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
            final HttpServletRequest httpRequest, final HttpServletResponse httpResponse) throws PayPalCustomerHasPayerIdException, PayPalDuplicateUidException {
        validate(user, "user", userSignUpDTOValidator);
        final RegisterData registerData = getDataMapper()
                .map(user, RegisterData.class, "login,password,titleCode,firstName,lastName");
        if (userService.isUserExisting(registerData.getLogin()) && payPalCustomerFacade.isPayerIdInCustomer(registerData.getLogin())) {
            throw new PayPalCustomerHasPayerIdException();
        }
        try {
            customerFacade.register(registerData);
        } catch (DuplicateUidException e) {
            LOG.debug("Duplicated UID", e);
            throw new PayPalDuplicateUidException(registerData.getLogin());
        }
        final String userId = user.getUid().toLowerCase(Locale.ENGLISH);
        httpResponse.setHeader(LOCATION, getAbsoluteLocationURL(httpRequest, userId));
        final CustomerData customerData = customerFacade.getUserForUID(userId);
        return getDataMapper().map(customerData, UserWsDTO.class, fields);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @RequestMapping(value = "/{userId}/getSetUpPasswordFormData", method = RequestMethod.GET)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @Operation(summary = "Get information about ability for setting single authentication password")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalSetUpPasswordFormWsData getSetUpPasswordFormData(
            @Parameter(description = "User identifier.", required = true) @PathVariable final String userId) {
        PayPalSetUpPasswordFormWsData setUpPasswordWsData = new PayPalSetUpPasswordFormWsData();
        final Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (containsRole(auth, "ROLE_TRUSTED_CLIENT") || containsRole(auth, "ROLE_CUSTOMERMANAGERGROUP")) {
            setUpPasswordWsData.setIsFormForSettingSingleAuthenticationPasswordActive(
                    defaultPayPalSetUpPasswordFacade.isSetUpPasswordForm(userId));
        } else {
            setUpPasswordWsData.setIsFormForSettingSingleAuthenticationPasswordActive(
                    defaultPayPalSetUpPasswordFacade.isSetUpPasswordForm());
        }
        return setUpPasswordWsData;
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP" })
    @RequestMapping(value = "/{userId}/unlink-paypal", method = RequestMethod.POST)
    @ResponseStatus(value = HttpStatus.OK)
    @Operation(summary = "Unlink login with PayPal for current customer")
    @ResponseBody
    @ApiBaseSiteIdAndUserIdParam
    public void unlinkLoginWithPayPal() {
        payPalCustomerFacade.unlinkLoginWithPayPal();
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @RequestMapping(value = "/{userId}/setPassword", method = RequestMethod.POST)
    @ResponseStatus(value = HttpStatus.ACCEPTED)
    @Operation(operationId = "setUpPassword", summary = "Set up a password for Single Auth", description = "Set up a password for Single Auth")
    @ApiBaseSiteIdAndUserIdParam
    public void setPasswordForSingleAuth(
            @Parameter(description = "User identifier.", required = true) @PathVariable final String userId,
            @Parameter(description = "New password.", required = true) @RequestParam final String newPassword) {
        final Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        final UserSignUpWsDTO customer = new UserSignUpWsDTO();
        customer.setPassword(newPassword);
        validate(customer, "password", passwordStrengthValidator);
        if (containsRole(auth, "ROLE_TRUSTED_CLIENT") || containsRole(auth, "ROLE_CUSTOMERMANAGERGROUP")) {
            customerFacade.setPassword(userId, newPassword);
        } else {
            defaultPayPalSetUpPasswordFacade.setUpPassword(newPassword);
        }
    }

    @Secured({"ROLE_CLIENT", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @SecurePortalUnauthenticatedAccess
    @RequestMapping(method = RequestMethod.POST)
    @ResponseStatus(value = HttpStatus.CREATED)
    @ResponseBody
    @RequestMappingOverride
    @Operation(hidden = true, summary = " Registers a customer", description =
            "Registers a customer. There are two options for registering a customer. The first option requires "
                    + "the following parameters: login, password, firstName, lastName, titleCode. The second option converts a guest to a customer. In this case, the required parameters are: guid, password.")
    @ApiBaseSiteIdParam
    @Parameter(name = CommercewebservicescommonsConstants.CAPTCHA_TOKEN_HEADER, description = CommercewebservicescommonsConstants.CAPTCHA_TOKEN_HEADER_DESC, required = false, schema = @Schema(type = "string"), in = ParameterIn.HEADER)
    @CaptchaAware
    public UserWsDTO createUser(
            @Parameter(description = "Customer's login. Customer login is case insensitive.") @RequestParam(required = false) final String login,
            @Parameter(description = "Customer's password.", required = true) @RequestParam final String password,
            @Parameter(description = "Customer's title code. For a list of codes, see /{baseSiteId}/titles resource") @RequestParam(required = false) final String titleCode,
            @Parameter(description = "Customer's first name.") @RequestParam(required = false) final String firstName,
            @Parameter(description = "Customer's last name.") @RequestParam(required = false) final String lastName,
            @Parameter(description = "Guest order's guid.") @RequestParam(required = false) final String guid,
            @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
            final HttpServletRequest httpRequest, final HttpServletResponse httpResponse) throws DuplicateUidException, PayPalGuestCustomerHasPayerIdException {
        final UserSignUpWsDTO user = new UserSignUpWsDTO();
        httpRequestUserSignUpDTOPopulator.populate(httpRequest, user);
        CustomerData customer = null;
        final String userId;
        if (guid != null) {
            validate(user, "user", guestConvertingDTOValidator);
            convertToCustomer(password, guid);
            customer = customerFacade.getCurrentCustomer();
            userId = customer.getUid();
        } else {
            validate(user, "user", userSignUpDTOValidator);
            registerNewUser(login, password, titleCode, firstName, lastName);
            userId = login.toLowerCase(Locale.ENGLISH);
            customer = customerFacade.getUserForUID(userId);
        }
        httpResponse.setHeader(LOCATION, getAbsoluteLocationURL(httpRequest, userId));
        return getDataMapper().map(customer, UserWsDTO.class, fields);
    }

    protected void convertToCustomer(final String password, final String guid) throws DuplicateUidException, PayPalGuestCustomerHasPayerIdException {
        LOG.debug("convertToCustomer: guid={}", sanitize(guid));

        try {
            customerFacade.changeGuestToCustomer(password, guid);
        } catch (final UnknownIdentifierException | IllegalArgumentException ex) {
            throw new RequestParameterException("Order with guid " + sanitize(guid) + " not found in current BaseStore",
                    RequestParameterException.UNKNOWN_IDENTIFIER, "guid", ex);
        } catch (DuplicateUidException ex) {
            String userEmail = payPalGuestCheckoutFacade.getCostumerEmailByGuid(guid);
            if (userService.isUserExisting(userEmail) && payPalCustomerFacade.isPayerIdInCustomer(userEmail)) {
                throw new PayPalGuestCustomerHasPayerIdException();
            } else {
                throw new DuplicateUidException();
            }
        }
    }

    protected void registerNewUser(final String login, final String password, final String titleCode, final String firstName,
                                   final String lastName) throws DuplicateUidException {
        LOG.debug("registerUser: login={}", sanitize(login));

        if (!EmailValidator.getInstance().isValid(login)) {
            throw new RequestParameterException("Login [" + sanitize(login) + "] is not a valid e-mail address!",
                    RequestParameterException.INVALID, "login");
        }

        final RegisterData registerData = createRegisterData(login, password, titleCode, firstName, lastName);
        customerFacade.register(registerData);
    }

    private RegisterData createRegisterData(final String login, final String password, final String titleCode,
                                            final String firstName, final String lastName) {
        final RegisterData registerData = new RegisterData();
        registerData.setFirstName(firstName);
        registerData.setLastName(lastName);
        registerData.setLogin(login);
        registerData.setPassword(password);
        registerData.setTitleCode(titleCode);
        return registerData;
    }

    protected String getAbsoluteLocationURL(final HttpServletRequest httpRequest, final String uid) {
        final String requestURL = httpRequest.getRequestURL().toString();
        final String encodedUid = UriUtils.encodePathSegment(uid, StandardCharsets.UTF_8.name());
        return UriComponentsBuilder.fromHttpUrl(requestURL).pathSegment(encodedUid).build().toString();
    }

    protected boolean containsRole(final Authentication auth, final String role) {
        for (final GrantedAuthority ga : auth.getAuthorities()) {
            if (ga.getAuthority().equals(role)) {
                return true;
            }
        }
        return false;
    }
}
