/*

 */
package com.paypalocc.controllers;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.core.model.PaypalAccessTokenModel;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PaypalAccessTokenService;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.data.ws.AddingPaymentMethodAndAddressStatusWsData;
import com.paypal.hybris.data.ws.PayPalCCSetupTokenWsDTO;
import com.paypal.hybris.data.ws.PayPalConnectLoginData;
import com.paypal.hybris.data.ws.PayPalConnectTokenWsData;
import com.paypal.hybris.data.ws.PayPalConnectWsData;
import com.paypal.hybris.data.ws.PayPalExpiredCardsMsgStatusWsData;
import com.paypal.hybris.data.ws.PayPalVaultForNewUserRequestWsDTO;
import com.paypal.hybris.data.ws.RegisterUserResponseWsDTO;
import com.paypal.hybris.facade.exception.PayPalUserIsDisabledException;
import com.paypal.hybris.facade.exception.PaypalAccessTokenNotFoundException;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalConnectWithPayPalNotificationFacade;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCreditCardFacade;
import com.paypalocc.populators.PaymentTokenDataPopulator;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.Map;
import java.util.Optional;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}/connect")
@CacheControl(directive = CacheControlDirective.NO_CACHE)
@Tag(name = "PayPal Connect")
public class PayPalConnectController extends PayPalBaseController {

    private static final Logger LOG = LoggerFactory.getLogger(PayPalConnectController.class);

    @Resource(name = "payPalCustomerFacade")
    private PayPalCustomerFacade payPalCustomerFacade;

    @Resource(name = "payPalCreditCardFacade")
    private PayPalCreditCardFacade payPalCreditCardFacade;

    @Resource(name = "payPalConnectService")
    private PayPalConnectService payPalConnectService;

    @Resource(name = "baseSiteService")
    private BaseSiteService baseSiteService;

    @Resource(name = "payPalCustomerAccountService")
    private PayPalCustomerAccountService payPalCustomerAccountService;

    @Resource(name = "payPalRegistrationUserFacade")
    private PayPalRegistrationUserFacade userFacade;

    @Resource(name = "defaultPayPalAcceleratorCheckoutFacade")
    private PayPalAcceleratorCheckoutFacade acceleratorCheckoutFacade;

    @Resource(name = "payPalConfigurationService")
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Resource(name = "defaultUserService")
    private UserService userService;

    @Resource
    private PaypalAccessTokenService paypalAccessTokenService;

    @Resource
    private PayPalConnectWithPayPalNotificationFacade payPalConnectWithPayPalNotificationFacade;

    @Resource
    private DefaultPayPalCreditCardFacade defaultPayPalCreditCardFacade;

    @Resource
    private PaymentTokenDataPopulator paymentTokenDataPopulator;


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/register")
    @ResponseStatus(HttpStatus.OK)
    @Operation(operationId = "register endpoint for connect with PayPal", summary = "register user by accessToken")
    @ApiBaseSiteIdAndUserIdParam
    public RegisterUserResponseWsDTO doRegister(
            @Parameter(description = "JSON object which contains access token", required = true)
            @RequestBody PayPalConnectTokenWsData payPalConnectTokenWsData)
            throws DuplicateUidException, PaypalAccessTokenNotFoundException {
        String accessToken = paypalAccessTokenService.getPaypalAccessToken(payPalConnectTokenWsData.getAccessTokenGuid())
                .map(PaypalAccessTokenModel::getAccessToken)
                .orElseThrow(PaypalAccessTokenNotFoundException::new);
        String payerId = userFacade.registerPayPalUser(accessToken);

        return GenericBuilder.of(RegisterUserResponseWsDTO::new)
                .with(RegisterUserResponseWsDTO::setPayerId, payerId)
                .build();
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/exchange")
    @ResponseStatus(HttpStatus.OK)
    @Operation(operationId = "exchange authorization code", summary = "exchange authorization code to access token")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalConnectWsData exchangeAuthorizationCode(
            @Parameter(description = "JSON object which contains authorization code", required = true)
            @RequestBody PayPalConnectTokenWsData payPalConnectTokenWsData) {
        final PayPalConnectWsData connectWsData = new PayPalConnectWsData();
        final String accessToken = payPalConnectService
                .exchangeAuthorizationCodeToAccessToken(payPalConnectTokenWsData.getAuthorizationCode());

        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);
        String email = userFacade.getEmailFromPayPalUserData(userData);
        if (userService.isUserExisting(email)) {
            connectWsData.setShouldSaveAddress(!payPalCustomerFacade.isPayPalAddressPresent(email, userData.getAddress()));
            connectWsData.setShouldSavePaymentInfo(!payPalCustomerFacade.isPayPalPaymentMethodPresent(email, userData));
            connectWsData.setIsRegistered(true);
            connectWsData.setEmail(email);
            connectWsData.setBaseSiteName(baseSiteService.getCurrentBaseSite().getName());
        } else {
            connectWsData.setShouldSaveAddress(true);
            connectWsData.setShouldSavePaymentInfo(defaultPayPalConfigurationService.isPayPalVaultFlowEnabled());
            connectWsData.setIsRegistered(false);
        }
        connectWsData.setIsUserAlreadyHaveAccount(payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id()));
        String accessTokenGuid = paypalAccessTokenService.savePaypalAccessToken(accessToken);
        connectWsData.setAccessTokenGuid(accessTokenGuid);
        connectWsData.setPayerId(userData.getPayer_id());
        return connectWsData;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/login")
    @ResponseStatus(HttpStatus.OK)
    @Operation(operationId = "login endpoint for connect with PayPal", summary = "Generate login data for user by accessToken")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalConnectLoginData login(
            @Parameter(description = "JSON object which contains accessToken", required = true)
            @RequestBody PayPalConnectTokenWsData payPalConnectTokenWsData) throws PayPalUserIsDisabledException, PaypalAccessTokenNotFoundException {
        String accessTokenGuid = payPalConnectTokenWsData.getAccessTokenGuid();
        String accessToken = paypalAccessTokenService.getPaypalAccessToken(accessTokenGuid)
                .map(PaypalAccessTokenModel::getAccessToken)
                .orElseThrow(PaypalAccessTokenNotFoundException::new);
        final PayPalConnectUserData userData = payPalConnectService
                .getUserDataByAccessToken(accessToken);
        final String email = userFacade.getEmailFromPayPalUserData(userData);
        if (!payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id())) {
            userFacade.setPayerIdToUser(email, userData.getPayer_id());
            payPalConnectWithPayPalNotificationFacade.sendLinkCustomerEmailNotification(email);
        }
        final CustomerData customer = payPalCustomerAccountService.getCustomerDataByUid(email);
        if (payPalCustomerFacade.isUserDisabled(customer.getDeactivationDate())) {
            throw new PayPalUserIsDisabledException();
        }
        final PayPalConnectLoginData loginData = new PayPalConnectLoginData();
        payPalCustomerAccountService.setAccessTokenForCustomer(accessTokenGuid, userData.getPayer_id());
        loginData.setLogin(customer.getUid());
        loginData.setAccessTokenGuid(accessTokenGuid);
        return loginData;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/afterLogin")
    @ResponseStatus(HttpStatus.OK)
    @Operation(operationId = "After login action", summary = "clear password field")
    @ApiBaseSiteIdAndUserIdParam
    public HttpStatus afterLogin(
            @Parameter(description = "JSON object which contains access token", required = true)
            @RequestBody PayPalConnectTokenWsData payPalConnectTokenWsData) throws PaypalAccessTokenNotFoundException {
        String accessTokenGuid = payPalConnectTokenWsData.getAccessTokenGuid();

        String accessToken = paypalAccessTokenService.getPaypalAccessToken(accessTokenGuid)
                .map(PaypalAccessTokenModel::getAccessToken)
                .orElseThrow(PaypalAccessTokenNotFoundException::new);
        PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);
        payPalCustomerAccountService.setDefaultAccessToken(userData.getPayer_id());

        paypalAccessTokenService.removePaypalAccessToken(accessTokenGuid);

        return HttpStatus.OK;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/payment-token")
    @ResponseStatus(HttpStatus.CREATED)
    @Operation(operationId = "addPayPalPayment", summary = "Save new PayPal payment info", description = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public AddingPaymentMethodAndAddressStatusWsData createPayPalPaymentAndSaveAddressForNewUser(
            @Parameter(description = "Request body parameter that contains" +
                    " billingAgreementToken and isSavePaymentInfo", required = true)
            @RequestBody final PayPalVaultForNewUserRequestWsDTO payPalVaultForNewUserRequestWsDTO)
            throws PaypalAccessTokenNotFoundException {
        final AddingPaymentMethodAndAddressStatusWsData result = new AddingPaymentMethodAndAddressStatusWsData();
        result.setIsAddressAdded(false);
        result.setIsPaymentMethodAdded(false);

        final String accessTokenGuid = payPalVaultForNewUserRequestWsDTO.getAccessTokenGuid();
        final String accessToken = paypalAccessTokenService.getPaypalAccessToken(accessTokenGuid)
                .map(PaypalAccessTokenModel::getAccessToken)
                .orElseThrow(PaypalAccessTokenNotFoundException::new);
        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);

        final String email = userFacade.getEmailFromPayPalUserData(userData);
        if (!payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id())) {
            userFacade.setPayerIdToUser(email, userData.getPayer_id());
            payPalConnectWithPayPalNotificationFacade.sendLinkCustomerEmailNotification(email);
        }

        if (Optional.of(payPalVaultForNewUserRequestWsDTO)
                .map(PayPalVaultForNewUserRequestWsDTO::getIsSavePaymentInfo)
                .map(Boolean::valueOf)
                .orElse(false)) {
            try {
                PaymentTokenData paymentTokenData = new PaymentTokenData();
                paymentTokenDataPopulator.populate(payPalVaultForNewUserRequestWsDTO.getVaultRequest(), paymentTokenData);
                paymentTokenData.setPayerId(userData.getPayer_id());
                defaultPayPalCreditCardFacade.requestPaymentToken(paymentTokenData);
                result.setIsPaymentMethodAdded(true);
            } catch (Exception ex) {
                LOG.error("Error during adding payment method for new customer", ex);
                result.setErrorMessagePaymentMethod(ex.getMessage());
            }
        }
        if (payPalVaultForNewUserRequestWsDTO.isIsSaveAddress() && userData.getAddress() != null) {
            try {
                acceleratorCheckoutFacade.addAddressForNewUser(userData.getAddress(), userData.getPayer_id());
                result.setIsAddressAdded(true);
            } catch (Exception ex) {
                LOG.error("Error during adding address for new customer", ex);
                result.setErrorMessageAddress(ex.getMessage());
            }
        }
        return result;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @GetMapping(value = "/getExpireStatus")
    @ResponseStatus(HttpStatus.OK)
    @Operation(operationId = "Get expire cards status", summary = "Get expire cards status for message")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalExpiredCardsMsgStatusWsData getExpireCardsStatusForMsg() {
        final PayPalExpiredCardsMsgStatusWsData payPalExpiredCardsMsgStatusWsData = new PayPalExpiredCardsMsgStatusWsData();
        payPalExpiredCardsMsgStatusWsData.setExpiredStatus(payPalCreditCardFacade.getCardsExpirationStatus());
        return payPalExpiredCardsMsgStatusWsData;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/setup-token")
    @ResponseStatus(HttpStatus.CREATED)
    @Operation(operationId = "create setup token token", summary = "Create new setup token", description = "Create and save new setup token")
    @ApiBaseSiteIdAndUserIdParam
    public PayPalCCSetupTokenWsDTO createSetupToken(@RequestBody final Map<String, String> body) {
        final String payerId = body.get("payerId");

        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL);
        setupTokenRequestData.setPayerId(payerId);

        CCSetupTokenData ccSetupTokenData = payPalCreditCardFacade.requestSetupToken(setupTokenRequestData);

        return getDataMapper().map(ccSetupTokenData, PayPalCCSetupTokenWsDTO.class);
    }

}
