package com.paypalocc.controllers;

import com.paypal.enums.PayPalPageType;
import com.paypal.hybris.data.PayPalFraudNetConfigurationData;
import com.paypal.hybris.data.PaypalGeneralPageConfigurationData;
import com.paypal.hybris.data.ws.PayPalFraudNetConfigurationWsDTO;
import com.paypal.hybris.data.ws.PaypalGeneralPageConfigurationWsDTO;
import com.paypal.hybris.facade.facades.PayPalConfigurationFacade;
import com.paypal.hybris.facade.strategy.configuration.page.PageConfigurationStrategy;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdUserIdAndCartIdParam;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Schema;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.List;
import java.util.Locale;

@RestController
@RequestMapping(value = "/{baseSiteId}")
public class PayPalConfigurationController extends PayPalBaseController {

    private static final String NO_HANDLER_FOUND_FOR_PAGE_TYPE_MESSAGE = "No handler found for %s page type";

    @Resource(name = "paypalConfigurationFacade")
    private PayPalConfigurationFacade payPalConfigurationFacade;

    @Resource(name = "pageConfigurationStrategies")
    private List<PageConfigurationStrategy> pageConfigurationStrategies;

    @GetMapping("/paypal/config/general")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdParam
    @Operation(summary = " Get the Paypal General Configuration for page")
    public PaypalGeneralPageConfigurationWsDTO getGeneralPageConfiguration(
            @Parameter(description = "Page type identifier.", schema = @Schema(allowableValues = {"minicart", "cart", "mark", "account, login"}),
                    required = true) @RequestParam final String pageType) {
        final PaypalGeneralPageConfigurationData payPalConfigurationData = selectPageConfigurationStrategy(PayPalPageType.valueOf(pageType.toUpperCase(Locale.US))).getConfiguration();
        return getDataMapper().map(payPalConfigurationData, PaypalGeneralPageConfigurationWsDTO.class);
    }

    @GetMapping("/users/{userId}/carts/{cartId}/paypal/config/fraud")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdUserIdAndCartIdParam
    @Operation(summary = " Get the Paypal Fraud Net Configuration")
                    public PayPalFraudNetConfigurationWsDTO getPayPalFraudConfiguration(
            @Parameter(description = "Page type identifier.", schema = @Schema(allowableValues = {"minicart", "cart", "mark", "account, login"}),
                    required = true) @RequestParam final String pageType) {
        final PayPalFraudNetConfigurationData payPalEssentialConfigurationData = payPalConfigurationFacade.getPayPalFraudNetConfigurationData(pageType);
        return getDataMapper().map(payPalEssentialConfigurationData, PayPalFraudNetConfigurationWsDTO.class);
    }

    private PageConfigurationStrategy selectPageConfigurationStrategy(PayPalPageType pageType) {
        return pageConfigurationStrategies.stream()
                .filter(strategy -> strategy.test(pageType))
                .findAny()
                .orElseThrow(() -> new IllegalArgumentException(String.format(NO_HANDLER_FOUND_FOR_PAGE_TYPE_MESSAGE, pageType)));
    }

}
