package com.paypal.hybris.b2baddon.security;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorstorefrontcommons.security.BruteForceAttackCounter;
import de.hybris.platform.b2b.model.B2BCustomerModel;
import de.hybris.platform.core.Constants;
import de.hybris.platform.core.model.user.UserGroupModel;
import de.hybris.platform.jalo.JaloSession;
import de.hybris.platform.jalo.user.User;
import de.hybris.platform.jalo.user.UserManager;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.spring.security.CoreAuthenticationProvider;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.MockitoAnnotations;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UserDetailsService;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalAuthenticationProviderTest {

    private static final String PRINCIPAL = "principal";
    private static final String NAME = "name";
    private static final String AUTH_DETAILS = "authDetails";
    @InjectMocks
    private PayPalAuthenticationProvider unit;
    @Mock
    private BruteForceAttackCounter bruteForceAttackCounter;
    @Mock
    private UserService userService;
    @Mock
    private UserDetailsService userDetailsService;
    @Mock
    private ModelService modelService;
    @Mock
    private Authentication authentication;
    @Mock
    private B2BCustomerModel b2BCustomerModel;
    @Mock
    private UserGroupModel userGroupModel;
    @Mock
    private UserDetails userDetails;
    @Mock
    private UserDetailsChecker userDetailsChecker;
    @Mock
    private UserManager userManager;
    @Mock
    private JaloSession jaloSession;
    @Mock
    private User userJalo;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(userService.getUserForUID(NAME)).thenReturn(b2BCustomerModel);
        when(authentication.getPrincipal()).thenReturn(PRINCIPAL);
        when(authentication.getName()).thenReturn(NAME);
        when(userService.getUserGroupForUID(Constants.USER.CUSTOMER_USERGROUP)).thenReturn(userGroupModel);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldBadCredentialsExceptionWhenLoginDisabled() {
        when(b2BCustomerModel.isLoginDisabled()).thenReturn(true);

        unit.authenticate(authentication);

        verify(bruteForceAttackCounter).resetUserCounter(NAME);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldBadCredentialsExceptionWhenBruteForceAttack() {
        when(bruteForceAttackCounter.isAttack(NAME)).thenReturn(true);

        unit.authenticate(authentication);

        verify(b2BCustomerModel).setLoginDisabled(true);
        verify(modelService).save(b2BCustomerModel);
        verify(bruteForceAttackCounter).resetUserCounter(NAME);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldBadCredentialsExceptionWhenUserHasNoCustomerGroupRole() {
        when(userService.isMemberOfGroup(b2BCustomerModel, userGroupModel)).thenReturn(Boolean.FALSE);

        unit.authenticate(authentication);
    }

    @Test
    public void shouldAuthenticate() {
        when(userDetails.getUsername()).thenReturn(NAME);
        when(userService.isMemberOfGroup(b2BCustomerModel, userGroupModel)).thenReturn(Boolean.TRUE);
        when(userDetailsService.loadUserByUsername(NAME)).thenReturn(userDetails);
        when(authentication.getDetails()).thenReturn(AUTH_DETAILS);

        try (MockedStatic<UserManager> userManagerMockedStatic = mockStatic(UserManager.class);
             MockedStatic<JaloSession> jaloSessionMockedStatic = mockStatic(JaloSession.class)) {
            userManagerMockedStatic.when(UserManager::getInstance).thenReturn(userManager);
            jaloSessionMockedStatic.when(JaloSession::getCurrentSession).thenReturn(jaloSession);

            when(userManager.getUserByLogin(NAME)).thenReturn(userJalo);

            Authentication authenticationResult = unit.authenticate(authentication);

            verify(jaloSession).setUser(userJalo);
            assertEquals(AUTH_DETAILS, authenticationResult.getDetails());
            assertEquals(NAME, authenticationResult.getPrincipal());
            assertEquals(NAME, authenticationResult.getName());
        }
    }

    @Test
    public void shouldSupportConnectWithPayPalAuthenticationToken() {
        assertTrue(unit.supports(ConnectWithPayPalAuthenticationToken.class));
    }

    @Test
    public void shouldNotSupportSuperClass() {
        assertFalse(unit.supports(CoreAuthenticationProvider.class));
    }

}
