package com.paypal.hybris.addon.security;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorstorefrontcommons.forms.UpdateEmailForm;
import de.hybris.platform.acceleratorstorefrontcommons.security.BruteForceAttackCounter;
import de.hybris.platform.core.Constants;
import de.hybris.platform.core.model.user.UserGroupModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;

import java.util.Collections;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalAuthenticationProviderTest {

    private static final String PRINCIPAL = "principal";
    private static final String NAME = "name";
    private static final String NONE_PROVIDED = "NONE_PROVIDED";
    private static final String NONE_PROVIDED_LOWER_CASE = "none_provided";
    private static final String UID = "uid";

    @Mock
    private BruteForceAttackCounter bruteForceAttackCounter;

    @Mock
    private UserService userService;

    @Mock
    private ModelService modelService;

    @Mock
    private Authentication authentication;

    @Mock
    private UserModel userModel;

    @Mock
    private UserDetails userDetails;

    @Mock
    private UserDetailsService userDetailsService;

    @Mock
    private UserGroupModel userGroupModel;

    @InjectMocks
    @Spy
    private PayPalAuthenticationProvider unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldSupportClass() {
        assertTrue(unit.supports(ConnectWithPayPalAuthenticationToken.class));
    }

    @Test
    public void shouldNotSupportClass() {
        assertFalse(unit.supports(UpdateEmailForm.class));
    }

    @Test
    public void shouldSuccessfullyAuthenticate() {
        ConnectWithPayPalAuthenticationToken result = new ConnectWithPayPalAuthenticationToken(NAME, Collections.emptyList());
        result.setDetails(PRINCIPAL);

        when(authentication.getPrincipal()).thenReturn(PRINCIPAL);
        when(authentication.getName()).thenReturn(NAME);
        when(authentication.getDetails()).thenReturn(PRINCIPAL);
        when(bruteForceAttackCounter.isAttack(NAME)).thenReturn(Boolean.FALSE);
        when(userService.getUserForUID(NAME)).thenReturn(userModel);
        when(userModel.isLoginDisabled()).thenReturn(Boolean.FALSE);
        when(userModel.getUid()).thenReturn(UID);
        when(userService.getUserGroupForUID(Constants.USER.CUSTOMER_USERGROUP)).thenReturn(userGroupModel);
        when(userService.isMemberOfGroup(userModel, userGroupModel)).thenReturn(Boolean.TRUE);
        when(userDetailsService.loadUserByUsername(NAME)).thenReturn(userDetails);
        when(userDetails.isAccountNonLocked()).thenReturn(Boolean.TRUE);
        when(userDetails.isEnabled()).thenReturn(Boolean.TRUE);
        when(userDetails.isAccountNonExpired()).thenReturn(Boolean.TRUE);
        when(userDetails.isCredentialsNonExpired()).thenReturn(Boolean.TRUE);
        when(userDetails.getAuthorities()).thenReturn(Collections.emptyList());

        doNothing().when(unit).setUserToJaloSession(userDetails);

        assertEquals(result, unit.authenticate(authentication));
    }
    @Test(expected = BadCredentialsException.class)
    public void shouldNotAuthenticateAndThrowBadCredentialsExceptionWhenLoginDisabled() {
        when(authentication.getPrincipal()).thenReturn(PRINCIPAL);
        when(authentication.getName()).thenReturn(NAME);
        when(bruteForceAttackCounter.isAttack(NAME)).thenReturn(Boolean.FALSE);
        when(userService.getUserForUID(NAME)).thenReturn(userModel);
        when(userModel.isLoginDisabled()).thenReturn(Boolean.TRUE);
        when(userModel.getUid()).thenReturn(UID);

        unit.authenticate(authentication);

        verify(bruteForceAttackCounter).resetUserCounter(UID);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldNotAuthenticateAndThrowBadCredentialsExceptionWhenBruteForceAttack() {
        when(authentication.getPrincipal()).thenReturn(null);
        when(bruteForceAttackCounter.isAttack(NONE_PROVIDED)).thenReturn(Boolean.TRUE);
        when(userService.getUserForUID(NONE_PROVIDED_LOWER_CASE)).thenReturn(userModel);
        when(userModel.isLoginDisabled()).thenReturn(Boolean.FALSE);
        when(userModel.getUid()).thenReturn(UID);

        unit.authenticate(authentication);

        verify(userModel).setLoginDisabled(Boolean.TRUE);
        verify(modelService).save(userModel);
        verify(bruteForceAttackCounter).resetUserCounter(UID);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldNotAuthenticateAndThrowBadCredentialsExceptionWhenUserIsNotAMemberOfUserGroup() {
        when(authentication.getPrincipal()).thenReturn(PRINCIPAL);
        when(authentication.getName()).thenReturn(NAME);
        when(bruteForceAttackCounter.isAttack(NAME)).thenReturn(Boolean.FALSE);
        when(userService.getUserForUID(NAME)).thenReturn(userModel);
        when(userModel.isLoginDisabled()).thenReturn(Boolean.FALSE);
        when(userModel.getUid()).thenReturn(UID);
        when(userService.getUserGroupForUID(Constants.USER.CUSTOMER_USERGROUP)).thenReturn(userGroupModel);
        when(userService.isMemberOfGroup(userModel, userGroupModel)).thenReturn(Boolean.FALSE);

        unit.authenticate(authentication);
    }
}