package com.paypal.hybris.addon.controllers.pages;

import com.paypal.hybris.addon.controllers.PaypaladdonControllerConstants;
import de.hybris.platform.acceleratorservices.enums.CheckoutPciOptionEnum;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.PreValidateCheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.PreValidateQuoteCheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.RequireHardLogIn;
import de.hybris.platform.acceleratorstorefrontcommons.checkout.steps.CheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.constants.WebConstants;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.pages.checkout.steps.AbstractCheckoutStepController;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.PlaceOrderForm;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.commercefacades.product.ProductOption;
import de.hybris.platform.commercefacades.product.data.ProductData;
import de.hybris.platform.commerceservices.order.CommerceCartModificationException;
import de.hybris.platform.payment.AdapterException;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import java.util.Arrays;

@Controller
@RequestMapping(value = "/paypal/checkout/multi/summary/")
public class PayPalLpmSummaryCheckoutStepController extends AbstractCheckoutStepController {

    private static final String CHECKOUT_AUTHORIZATION_FAILED = "checkout.error.authorization.failed";
    private static final String CHECKOUT_PLACE_ORDER_FAILED = "checkout.placeOrder.failed";
    private static final String CHECKOUT_DELIVERY_ADDRESS_NOT_SELECTED = "checkout.deliveryAddress.notSelected";
    private static final String CHECKOUT_DELIVERY_METHOD_NOT_SELECTED = "checkout.deliveryMethod.notSelected";
    private static final String CHECKOUT_PAYMENT_METHOD_NOT_SELECTED = "checkout.paymentMethod.notSelected";
    private static final String CHECKOUT_TAX_MISSING = "checkout.error.tax.missing";
    private static final String CHECKOUT_CART_NOT_CALCULATED = "checkout.error.tax.missing";
    private static final String SUMMARY = "summary";

    private static final Logger LOG = Logger.getLogger(PayPalLpmSummaryCheckoutStepController.class);

    @RequestMapping(value = "/placeOrder")
    @PreValidateQuoteCheckoutStep
    @RequireHardLogIn
    public String placeOrder(@ModelAttribute("placeOrderForm") final PlaceOrderForm placeOrderForm, final Model model,
        final HttpServletRequest request, final RedirectAttributes redirectModel)
        throws CMSItemNotFoundException,
        CommerceCartModificationException {
        if (validateOrderForm(model)) {
            return enterStep(model, redirectModel);
        }
        if (validateCart(redirectModel)) {
            return REDIRECT_PREFIX + "/cart";
        }
        boolean isPaymentAuthorized = false;
        try {
            isPaymentAuthorized = getCheckoutFacade().authorizePayment(null);
        } catch (final AdapterException ae) {
            LOG.error(ae.getMessage(), ae);
        }
        if (!isPaymentAuthorized) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_AUTHORIZATION_FAILED);
            return enterStep(model, redirectModel);
        }
        final OrderData orderData;
        try {
            orderData = getCheckoutFacade().placeOrder();
        } catch (final Exception e) {
            LOG.error("Failed to place Order", e);
            GlobalMessages.addErrorMessage(model, CHECKOUT_PLACE_ORDER_FAILED);
            return enterStep(model, redirectModel);
        }
        return redirectToOrderConfirmationPage(orderData);
    }

    protected boolean validateOrderForm(final Model model) {
        boolean invalid = false;
        if (getCheckoutFlowFacade().hasNoDeliveryAddress()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_DELIVERY_ADDRESS_NOT_SELECTED);
            invalid = true;
        }
        if (getCheckoutFlowFacade().hasNoDeliveryMode()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_DELIVERY_METHOD_NOT_SELECTED);
            invalid = true;
        }
        if (getCheckoutFlowFacade().hasNoPaymentInfo()) {
            GlobalMessages.addErrorMessage(model, CHECKOUT_PAYMENT_METHOD_NOT_SELECTED);
            invalid = true;
        }
        final CartData cartData = getCheckoutFacade().getCheckoutCart();
        if (!getCheckoutFacade().containsTaxValues()) {
            LOG.error(String.format(
                "Cart %s does not have any tax values, which means the tax cacluation was not properly done, placement of order can't continue",
                cartData.getCode()));
            GlobalMessages.addErrorMessage(model, CHECKOUT_TAX_MISSING);
            invalid = true;
        }
        if (!cartData.isCalculated()) {
            LOG.error(
                String.format("Cart %s has a calculated flag of FALSE, placement of order can't continue",
                    cartData.getCode()));
            GlobalMessages.addErrorMessage(model, CHECKOUT_CART_NOT_CALCULATED);
            invalid = true;
        }
        return invalid;
    }

    @RequestMapping(value = "/view", method = RequestMethod.GET)
    @RequireHardLogIn
    @Override
    @PreValidateQuoteCheckoutStep
    @PreValidateCheckoutStep(checkoutStep = SUMMARY)
    public String enterStep(final Model model, final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException, CommerceCartModificationException
    {
        final CartData cartData = getCheckoutFacade().getCheckoutCart();
        if (cartData.getEntries() != null && !cartData.getEntries().isEmpty())
        {
            for (final OrderEntryData entry : cartData.getEntries())
            {
                final String productCode = entry.getProduct().getCode();
                final ProductData product = getProductFacade().getProductForCodeAndOptions(productCode, Arrays.asList(
                        ProductOption.BASIC, ProductOption.PRICE, ProductOption.VARIANT_MATRIX_BASE, ProductOption.PRICE_RANGE));
                entry.setProduct(product);
            }
        }

        model.addAttribute("cartData", cartData);
        model.addAttribute("allItems", cartData.getEntries());
        model.addAttribute("deliveryAddress", cartData.getDeliveryAddress());
        model.addAttribute("deliveryMode", cartData.getDeliveryMode());
        model.addAttribute("paymentInfo", cartData.getPaymentInfo());

        // Only request the security code if the SubscriptionPciOption is set to Default.
        final boolean requestSecurityCode = CheckoutPciOptionEnum.DEFAULT
                .equals(getCheckoutFlowFacade().getSubscriptionPciOption());
        model.addAttribute("requestSecurityCode", Boolean.valueOf(requestSecurityCode));

        model.addAttribute(new PlaceOrderForm());

        final ContentPageModel multiCheckoutSummaryPage = getContentPageForLabelOrId(MULTI_CHECKOUT_SUMMARY_CMS_PAGE_LABEL);
        storeCmsPageInModel(model, multiCheckoutSummaryPage);
        setUpMetaDataForContentPage(model, multiCheckoutSummaryPage);

        model.addAttribute(WebConstants.BREADCRUMBS_KEY,
                getResourceBreadcrumbBuilder().getBreadcrumbs("checkout.multi.summary.breadcrumb"));
        model.addAttribute("metaRobots", "noindex,nofollow");
        setCheckoutStepLinksForModel(model, getCheckoutStep());
        return PaypaladdonControllerConstants.Views.Pages.Checkout.CHECKOUT_SUMMARY_PAGE;
    }

    @RequestMapping(value = "/back", method = RequestMethod.GET)
    @RequireHardLogIn
    @Override
    public String back(final RedirectAttributes redirectAttributes)
    {
        return getCheckoutStep().previousStep();
    }

    @RequestMapping(value = "/next", method = RequestMethod.GET)
    @RequireHardLogIn
    @Override
    public String next(final RedirectAttributes redirectAttributes)
    {
        return getCheckoutStep().nextStep();
    }

    protected CheckoutStep getCheckoutStep()
    {
        return getCheckoutStep(SUMMARY);
    }
}
