/**
 *
 */
package com.paypal.hybris.addon.controllers.pages;

import com.paypal.hybris.addon.constants.PaypaladdonWebConstants;
import com.paypal.hybris.addon.controllers.PaypaladdonControllerConstants;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.addon.security.PayPalAutoLoginStrategy;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.exception.PayPalConnectException;
import com.paypal.hybris.core.exception.PayPalUnverifiedAccountException;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalConnectWithPayPalNotificationFacade;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.platform.acceleratorfacades.flow.CheckoutFlowFacade;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.pages.AbstractLoginPageController;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.GuestForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.LoginForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.RegisterForm;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.AbstractPageModel;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.security.web.savedrequest.HttpSessionRequestCache;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;
import org.springframework.web.servlet.support.RequestContextUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.CONNECT_LOGIN_ACCOUNT_CONNECT;
import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.CONNECT_LOGIN_ACCOUNT_CONNECT_PASSWORD_VERIFICATION_PAGE;
import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.DISABLE_LOGIN_VIA_PAYPAL_TO_THE_EXISTING_USER_MESSAGE;
import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.HOME_PAGE;
import static com.paypal.hybris.addon.constants.PaypaladdonWebConstants.REDIRECT;
import static com.paypal.hybris.addon.controllers.PaypaladdonControllerConstants.Views.Pages.Login.ACCOUNT_CONNECT_PASSWORD_VERIFICATION_PAGE;
import static com.paypal.hybris.addon.controllers.PaypaladdonControllerConstants.Views.Pages.Login.SAVE_PAYMENT_INFO_PAGE;
import static de.hybris.platform.util.localization.Localization.getLocalizedString;
import static java.util.Objects.nonNull;


@Controller
@RequestMapping(value = "/connect/login")
public class PayPalConnectLoginController extends AbstractLoginPageController {

    private static final Logger LOG = Logger.getLogger(PayPalConnectLoginController.class);

    private static final String PAYPAL_CONNECT_GLOBAL_ERROR = "paypal.connect.registration.global.error";
    private static final String PAYPAL_CONNECT_SERVLET_PATH = "/checkout";
    private static final String PAYPAL_PAGE_INDICATOR = "isLoginPage";
    private static final String PAYER_ID = "payerId";
    private static final String LOGIN_PAGE_REDIRECT = REDIRECT + "/login";
    private static final String LOGIN_CHECKOUT_PAGE_REDIRECT = LOGIN_PAGE_REDIRECT + "/checkout";
    private static final String ADDRESS_DATA = "addressData";
    private static final String IS_REQUEST_FROM_CHECKOUT_PAGE = "isRequestFromCheckoutPage";
    private static final String SAVE_PAYMENT_INFO = "savePaymentInfoUrl";
    private static final String SAVE_PAYMENT_INFO_URL = "connect/login/savePaymentInfo";
    private static final String SAVE_PAYMENT_INFO_CHECKOUT_URL = "connect/login/savePaymentInfo/checkout";
    private static final String SHOULD_SAVE_ADDRESS = "shouldSaveAddress";
    private static final String SHOULD_SAVE_PAYMENT_INFO = "shouldSavePaymentInfo";
    private static final String WAS_USER_REGISTERED = "wasUserRegistered";
    private static final String DID_USER_HAVE_PAYER_ID = "didCustomerHavePayerId";
    private static final String UNVERIFIED_ACCOUNT_ERROR = "paypal.connect.unverified.account.error";
    private static final String PASSWORD_VERIFICATION_ERROR = "paypal.connect.password.verification.error";
    private static final String PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS = "paypal.connect.login.linked.accounts";
    private static final String EMAIL_ATTRIBUTE = "email";
    private static final String USER_DATA_ATTRIBUTE = "userData";
    private static final String BASE_SITE_NAME_ATTRIBUTE = "baseSiteName";
    private static final String VAULT_PROCESS = "vaultProcessUrl";
    private static final String VAULT_PROCESS_URL = "connect/login/response";

    private HttpSessionRequestCache httpSessionRequestCache;

    @Resource(name = "payPalCustomerFacade")
    private PayPalCustomerFacade payPalCustomerFacade;

    @Resource(name = "payPalConnectService")
    private PayPalConnectService payPalConnectService;

    @Resource(name = "payPalConfigurationService")
    private PayPalConfigurationService payPalConfigurationService;

    @Resource(name = "payPalCustomerAccountService")
    private PayPalCustomerAccountService payPalCustomerAccountService;

    @Resource(name = "payPalRegistrationUserFacade")
    private PayPalRegistrationUserFacade userFacade;

    @Resource(name = "checkoutFlowFacade")
    private CheckoutFlowFacade checkoutFlowFacade;

    @Resource(name = "sessionService")
    private SessionService sessionService;

    @Resource(name = "defaultPayPalAcceleratorCheckoutFacade")
    private PayPalAcceleratorCheckoutFacade acceleratorCheckoutFacade;

    @Resource(name = "payPalCreditCardFacade")
    private PayPalCreditCardFacade payPalCreditCardFacade;

    @Resource(name = "payPalAutoLoginStrategy")
    private PayPalAutoLoginStrategy payPalAutoLoginStrategy;

    @Resource(name = "defaultUserService")
    private UserService userService;

    @Resource
    private PayPalConnectWithPayPalNotificationFacade payPalConnectWithPayPalNotificationFacade;

    @Resource
    private PayPalPaymentInfoService payPalPaymentInfoService;

    @Resource(name = "payPalConfigurationService")
    private PayPalConfigurationService defaultConfigurationService;

    @Override
    protected String getView() {
        final boolean isLoginPage = sessionService.getAttribute(PAYPAL_PAGE_INDICATOR);
        sessionService.removeAttribute(PAYPAL_PAGE_INDICATOR);
        if (isLoginPage) {
            return PaypaladdonControllerConstants.Views.Pages.Account.ACCOUNT_LOGIN_PAGE;
        }
        return PaypaladdonControllerConstants.Views.Pages.Checkout.CHECKOUT_LOGIN_PAGE;
    }

    @Override
    protected String getSuccessRedirect(final HttpServletRequest request, final HttpServletResponse response) {
        if (isRequestFromCheckoutPage(request)) {
            if (hasItemsInCart()) {
                return getCheckoutUrl();
            }
            return PAYPAL_CONNECT_SERVLET_PATH;
        }
        if (httpSessionRequestCache.getRequest(request, response) != null) {
            return httpSessionRequestCache.getRequest(request, response).getRedirectUrl();
        }
        return "/";
    }

    private boolean isRequestFromCheckoutPage(final HttpServletRequest request) {
        return request.getServletPath().contains(PAYPAL_CONNECT_SERVLET_PATH);
    }

    @Override
    protected AbstractPageModel getCmsPage() throws CMSItemNotFoundException {
        final boolean isLoginPage = sessionService.getAttribute(PAYPAL_PAGE_INDICATOR);
        if (isLoginPage) {
            return getContentPageForLabelOrId("login");
        }
        return getContentPageForLabelOrId("checkout-login");
    }

    private String handleRegistrationError(final Model model, final HttpServletRequest request) throws CMSItemNotFoundException {
        String view;
        ContentPageModel pageModel;
        if (isRequestFromCheckoutPage(request)) {
            pageModel = getContentPageForLabelOrId("checkout-login");
            view = PaypaladdonControllerConstants.Views.Pages.Checkout.CHECKOUT_LOGIN_PAGE;
        } else {
            pageModel = getContentPageForLabelOrId("login");
            view = PaypaladdonControllerConstants.Views.Pages.Account.ACCOUNT_LOGIN_PAGE;
        }
        storeCmsPageInModel(model, pageModel);
        setUpMetaDataForContentPage(model, pageModel);
        addRegistrationConsentDataToModel(model);
        return view;
    }


    @Resource(name = "httpSessionRequestCache")
    public void setHttpSessionRequestCache(final HttpSessionRequestCache accHttpSessionRequestCache) {
        this.httpSessionRequestCache = accHttpSessionRequestCache;
    }

    @RequestMapping(method = RequestMethod.GET)
    public String doLogin(@RequestHeader(value = "referer", required = false) final String referer,
                          final RegisterForm form, final Model model,
                          final HttpServletRequest request, final HttpServletResponse response,
                          @RequestParam(required = false) String code, HttpSession session,
                          final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        sessionService.setAttribute(PAYPAL_PAGE_INDICATOR, true);
        if (code == null) {
            storeReferer(referer, request, response);
            return getDefaultLoginPage(false, session, model);
        }
        final String accessToken = payPalConnectService.exchangeAuthorizationCodeToAccessToken(code);
        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);
        storeReferer(referer, request, response);
        return processRequest(model, form, request, response, accessToken, userData, redirectAttributes);
    }

    @RequestMapping(value = "/checkout", method = RequestMethod.GET)
    public String doCheckoutLogin(final Model model, final RegisterForm form, final HttpServletRequest request,
                                  final HttpServletResponse response, @RequestParam(required = false) final String code, HttpSession session,
                                  final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        sessionService.setAttribute(PAYPAL_PAGE_INDICATOR, false);
        if (code == null) {
            model.addAttribute("expressCheckoutAllowed", Boolean.valueOf(checkoutFlowFacade.isExpressCheckoutEnabledForStore()));
            return getDefaultLoginPage(false, session, model);
        }
        final String accessToken = payPalConnectService.exchangeAuthorizationCodeToAccessToken(code);
        final PayPalConnectUserData userData = payPalConnectService.getUserDataByAccessToken(accessToken);
        model.addAttribute("expressCheckoutAllowed", Boolean.valueOf(checkoutFlowFacade.isExpressCheckoutEnabledForStore()));
        return processRequest(model, form, request, response, accessToken, userData, redirectAttributes);
    }

    private String processRequest(Model model, RegisterForm form, HttpServletRequest request, HttpServletResponse response,
                                  String accessToken, PayPalConnectUserData userData, RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        String email;
        try {
            email = userFacade.getEmailFromPayPalUserData(userData);
        } catch (PayPalUnverifiedAccountException e) {
            LOG.error(e.getMessage(), e);
            GlobalMessages.addFlashMessage(redirectAttributes,
                    GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString(UNVERIFIED_ACCOUNT_ERROR));
            return LOGIN_PAGE_REDIRECT;
        }

        if (userService.isUserExisting(email)) {
            if (!payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id())) {
                if (!payPalConfigurationService.isEnableLoginViaPayPalToTheExistingUser()) {
                    GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                            GlobalMessages.ERROR_MESSAGES_HOLDER, DISABLE_LOGIN_VIA_PAYPAL_TO_THE_EXISTING_USER_MESSAGE, null);
                    return LOGIN_PAGE_REDIRECT;
                }
                sessionService.setAttribute(USER_DATA_ATTRIBUTE, userData);
                sessionService.setAttribute(EMAIL_ATTRIBUTE, email);
                if (payPalConfigurationService.isPayPalAdditionalSecurityEnabled()) {
                    return REDIRECT + CONNECT_LOGIN_ACCOUNT_CONNECT_PASSWORD_VERIFICATION_PAGE;
                } else {
                    return REDIRECT + CONNECT_LOGIN_ACCOUNT_CONNECT;
                }
            }
            return processLogin(request, response, userData.getPayer_id());
        } else {
            try {
                userFacade.registerPayPalUser(accessToken);
                userFacade.setPayerIdToUser(email, userData.getPayer_id());
                boolean isShouldAddPaymentMethod = defaultConfigurationService.isPayPalVaultEnabled() &&
                        defaultConfigurationService.isPayPalEnable();
                setSessionAttributeForGlobalMessageToRegisteredAndNotRegisteredUser(false, false);
                if (payPalConfigurationService.isPayPalConnectAddPaymentMethodFlow()) {
                    setSessionAttributeForPaymentAndAddressInfo(true, isShouldAddPaymentMethod, userData, request);
                    return REDIRECT + CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
                } else {
                    displayingGlobalMessageToRegisteredAndNotRegisteredUser(redirectAttributes);
                    return processLogin(request, response, userData.getPayer_id());
                }
            } catch (PayPalUnverifiedAccountException e) {
                LOG.error(e.getMessage(), e);
                GlobalMessages.addFlashMessage(redirectAttributes,
                        GlobalMessages.ERROR_MESSAGES_HOLDER,
                        getLocalizedString(UNVERIFIED_ACCOUNT_ERROR));
                return LOGIN_PAGE_REDIRECT;
            } catch (DuplicateUidException e) {
                form.setTermsCheck(false);
                model.addAttribute(form);
                model.addAttribute(new LoginForm());
                model.addAttribute(new GuestForm());
                GlobalMessages.addErrorMessage(model, PAYPAL_CONNECT_GLOBAL_ERROR);
                return handleRegistrationError(model, request);
            }
        }
    }

    private String processLogin(final HttpServletRequest request, final HttpServletResponse response,
                                final String payerId) {
        final CustomerData customer = payPalCustomerAccountService.getCustomerDataByPayerId(payerId);
        if (payPalCustomerFacade.isUserDisabled(customer.getDeactivationDate())) {
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.ERROR_MESSAGES_HOLDER, "paypal.connect.login.customer.deactivated", null);
            return LOGIN_PAGE_REDIRECT;
        }

        String expiredStatus = payPalCreditCardFacade.getCardsExpirationStatus(customer.getUid());
        if (expiredStatus.equals(ExpirationStatus.EXPIRED.getCode())) {
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.CONF_MESSAGES_HOLDER, PaypaladdonWebConstants.EXPIRED_MESSAGE, null);
        }
        if (expiredStatus.equals(ExpirationStatus.EXPIRE_SOON.getCode())) {
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.CONF_MESSAGES_HOLDER, PaypaladdonWebConstants.EXPIRED_SOON_MESSAGE, null);
        }

        payPalAutoLoginStrategy.login(customer.getUid(), request, response);
        return "redirect:" + this.getSuccessRedirect(request, response);
    }

    @RequestMapping(value = "/savePaymentInfoAndAddressPage", method = RequestMethod.GET)
    public String getSavePaymentInfoAndAddressPage(final Model model, HttpServletRequest request) throws CMSItemNotFoundException {
        setModelAttributesForPaymentAndAddressInfo(request, model);
        storeCmsPageInModel(model, getCmsPage());
        setUpMetaDataForContentPage(model, (ContentPageModel) getCmsPage());
        return SAVE_PAYMENT_INFO_PAGE;
    }

    @PostMapping(value = {"/savePaymentInfo", "/savePaymentInfo/checkout"})
    public String doSavePaymentInfo(final RegisterForm form, final HttpServletRequest request,
                                    final HttpServletResponse response,
                                    final Model model, final RedirectAttributes redirectAttributes) {
        model.addAttribute(form);
        final String payerId = sessionService.getAttribute(PAYER_ID);
        displayingGlobalMessageToRegisteredAndNotRegisteredUser(redirectAttributes);
        removeSessionAttribute();
        return processLogin(request, response, payerId);
    }

    @RequestMapping(value = "/accountConnectPasswordVerificationPage", method = RequestMethod.GET)
    public String getAccountConnectVerificationPage(final Model model, @RequestParam(required = false) final boolean invalidPassword)
            throws CMSItemNotFoundException {
        PayPalConnectUserData userData = sessionService.getAttribute(USER_DATA_ATTRIBUTE);
        if (nonNull(userData) && !payPalCustomerAccountService.isCustomerWithPayerIdExist(userData.getPayer_id())) {
            model.addAttribute(EMAIL_ATTRIBUTE, sessionService.getAttribute(EMAIL_ATTRIBUTE));
            model.addAttribute(BASE_SITE_NAME_ATTRIBUTE, getBaseSiteService().getCurrentBaseSite().getName());
            storeCmsPageInModel(model, getCmsPage());
            setUpMetaDataForContentPage(model, (ContentPageModel) getCmsPage());
            if (invalidPassword) {
                GlobalMessages.addErrorMessage(model, getLocalizedString(PASSWORD_VERIFICATION_ERROR));
            }
            return ACCOUNT_CONNECT_PASSWORD_VERIFICATION_PAGE;
        }
        return REDIRECT + HOME_PAGE;
    }

    @RequestMapping(value = "/account-connect", method = RequestMethod.GET)
    public String doConnectPayPalAccount(final HttpServletRequest request, final HttpServletResponse response) {
        PayPalConnectUserData userData = sessionService.getAttribute(USER_DATA_ATTRIBUTE);
        String email = sessionService.getAttribute(EMAIL_ATTRIBUTE);
        userFacade.setPayerIdToUser(email, userData.getPayer_id());
        payPalConnectWithPayPalNotificationFacade.sendLinkCustomerEmailNotification(email);
        if (!payPalConfigurationService.isPayPalConnectAddPaymentMethodFlow()) {
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.CONF_MESSAGES_HOLDER, PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS, null);
            return processLogin(request, response, userData.getPayer_id());
        }
        boolean isPayPalAddressPresent = payPalCustomerFacade.isPayPalAddressPresent(email, userData.getAddress());
        boolean isPayPalPaymentMethodPresent = payPalCustomerFacade.isPayPalPaymentMethodPresent(email, userData);
        if (isPayPalPaymentMethodPresent && isPayPalAddressPresent) {
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.CONF_MESSAGES_HOLDER, PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS, null);
            return processLogin(request, response, userData.getPayer_id());
        }
        setSessionAttributeForPaymentAndAddressInfo(
                !isPayPalAddressPresent, !isPayPalPaymentMethodPresent, userData, request);
        setSessionAttributeForGlobalMessageToRegisteredAndNotRegisteredUser(true, false);
        sessionService.removeAttribute(USER_DATA_ATTRIBUTE);
        sessionService.removeAttribute(EMAIL_ATTRIBUTE);
        return REDIRECT + CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
    }

    @PostMapping(value = {"/response", "/response/checkout"})
    public String doHandlePayPalResponse(final PaymentTokenData paymentTokenData,
                                         final HttpServletRequest request, final HttpServletResponse response,
                                         final RedirectAttributes redirectAttributes) {
        final String payerId = sessionService.getAttribute(PAYER_ID);
        final PayPalConnectAddressData addressData = sessionService.getAttribute(ADDRESS_DATA);
        if ((boolean) sessionService.getAttribute(SHOULD_SAVE_PAYMENT_INFO)) {
            try {
                payPalCreditCardFacade.requestPaymentToken(paymentTokenData);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER, "paypal.connect.saved.payment.method");
            } catch (Exception e) {
                LOG.error("Error during adding payment method for new customer", e);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER, "paypal.connect.saved.payment.method.error");
            }
        }
        if ((boolean) sessionService.getAttribute(SHOULD_SAVE_ADDRESS)) {
            try {
                if (addressData != null) {
                    acceleratorCheckoutFacade.addAddressForNewUser(addressData, payerId);
                    GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER, "paypal.connect.saved.address");
                }
            } catch (Exception e) {
                LOG.error("Error during adding address for new customer", e);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER, "paypal.connect.saved.address.error");
            }
        }
        displayingGlobalMessageToRegisteredAndNotRegisteredUser(redirectAttributes);
        removeSessionAttribute();
        return processLogin(request, response, payerId);
    }

    @ExceptionHandler(PayPalConnectException.class)
    public String handlerPayPalConnectException(HttpServletRequest request, PayPalConnectException e) {
        GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                GlobalMessages.ERROR_MESSAGES_HOLDER, "paypal.connect.access.error", null);
        return isRequestFromCheckoutPage(request) ? LOGIN_CHECKOUT_PAGE_REDIRECT : LOGIN_PAGE_REDIRECT;
    }

    protected void storeReferer(final String referer, final HttpServletRequest request,
                                final HttpServletResponse response) {
        if (StringUtils.isNotBlank(referer) && !StringUtils.endsWith(referer, "/connect/login")
                && referer.startsWith(request.getServerName())) {
            httpSessionRequestCache.saveRequest(request, response);
        }
    }

    private boolean hasItemsInCart() {
        final CartData cartData = getCheckoutFlowFacade().getCheckoutCart();

        return cartData.getEntries() != null && !cartData.getEntries().isEmpty();
    }

    private static String getCheckoutUrl() {
        return "/checkout/multi/delivery-address/add";
    }

    protected CheckoutFlowFacade getCheckoutFlowFacade() {
        return checkoutFlowFacade;
    }

    private void setModelAttributesForPaymentAndAddressInfo(HttpServletRequest request, Model model) {
        model.addAttribute(SHOULD_SAVE_ADDRESS, sessionService.getAttribute(SHOULD_SAVE_ADDRESS));
        model.addAttribute(SHOULD_SAVE_PAYMENT_INFO, sessionService.getAttribute(SHOULD_SAVE_PAYMENT_INFO));
        if ((boolean) sessionService.getAttribute(IS_REQUEST_FROM_CHECKOUT_PAGE)) {
            model.addAttribute(SAVE_PAYMENT_INFO, SAVE_PAYMENT_INFO_CHECKOUT_URL);
        } else {
            model.addAttribute(SAVE_PAYMENT_INFO, SAVE_PAYMENT_INFO_URL);
            model.addAttribute(VAULT_PROCESS, VAULT_PROCESS_URL);
        }
    }

    private void setSessionAttributeForPaymentAndAddressInfo(boolean shouldSavaAddress,
                                                             boolean shouldSavePaymentInfo,
                                                             final PayPalConnectUserData userData,
                                                             final HttpServletRequest request) {

        sessionService.setAttribute(SHOULD_SAVE_ADDRESS, shouldSavaAddress);
        sessionService.setAttribute(SHOULD_SAVE_PAYMENT_INFO, shouldSavePaymentInfo);
        sessionService.setAttribute(PAYER_ID, userData.getPayer_id());
        sessionService.setAttribute(ADDRESS_DATA, userData.getAddress());
        sessionService.setAttribute(IS_REQUEST_FROM_CHECKOUT_PAGE, isRequestFromCheckoutPage(request));
    }

    private void setSessionAttributeForGlobalMessageToRegisteredAndNotRegisteredUser(boolean wasUserRegistered, boolean didUserHavePayerId) {
        sessionService.setAttribute(WAS_USER_REGISTERED, wasUserRegistered);
        sessionService.setAttribute(DID_USER_HAVE_PAYER_ID, didUserHavePayerId);
    }

    private void displayingGlobalMessageToRegisteredAndNotRegisteredUser(RedirectAttributes redirectAttributes) {
        if (nonNull(sessionService.getAttribute(WAS_USER_REGISTERED))
                && nonNull(sessionService.getAttribute(DID_USER_HAVE_PAYER_ID))) {
            boolean wasUserRegistered = sessionService.getAttribute(WAS_USER_REGISTERED);
            boolean didUserHavePayerId = sessionService.getAttribute(DID_USER_HAVE_PAYER_ID);
            if (!wasUserRegistered) {
                GlobalMessages.addFlashMessage(redirectAttributes,
                        GlobalMessages.CONF_MESSAGES_HOLDER, "paypal.connect.registration.message.set.up.password.for.new.user", null);
            } else if (!didUserHavePayerId) {
                GlobalMessages.addFlashMessage(redirectAttributes,
                        GlobalMessages.CONF_MESSAGES_HOLDER, PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS, null);
            }
        }
        sessionService.removeAttribute(WAS_USER_REGISTERED);
        sessionService.removeAttribute(DID_USER_HAVE_PAYER_ID);
    }

    private void removeSessionAttribute() {
        sessionService.removeAttribute(PAYER_ID);
        sessionService.removeAttribute(ADDRESS_DATA);
        sessionService.removeAttribute(SHOULD_SAVE_ADDRESS);
        sessionService.removeAttribute(SHOULD_SAVE_PAYMENT_INFO);
        sessionService.removeAttribute(IS_REQUEST_FROM_CHECKOUT_PAGE);
    }

    private PaymentTokenData preparePaypalTokenDataForRequest(CCSetupTokenData ccSetupTokenData) {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setSetupToken(ccSetupTokenData.getId());
        paymentTokenData.setAddressId(ccSetupTokenData.getSelectedBillingAddress());
        paymentTokenData.setPaymentInfoPK(ccSetupTokenData.getPaymentInfoPK());

        return paymentTokenData;
    }
}

