package com.paypalocc.validators;

import com.paypal.hybris.core.util.ExpiredDateValidatorUtil;
import com.paypal.hybris.data.ws.CreditCardAddressWsDTO;
import de.hybris.bootstrap.annotations.UnitTest;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.validation.Errors;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;

@UnitTest
public class PaypalExpiredDateDtoValidatorTest {

	private static final String EXPIRY_YEAR = "expiryYear";
	private static final String EXPIRY_MONTH = "expiryMonth";

	private static final long MONTH_TO_EXPIRATION = 5L;
	private static final long MONTH_TO_EXPIRATION_NEGATIVE = -5L;
	private static final String CREDIT_CARD_EXPIRY_MONTH_INVALID = "credit.card.expiry.month.invalid";
	private static final String CREDIT_CARD_EXPIRY_YEAR_INVALID = "credit.card.expiry.year.invalid";

	@Mock
	private ExpiredDateValidatorUtil expiredDateValidatorUtil;
	@Mock
	private Errors errors;
	@InjectMocks
	private PaypalExpiredDateDtoValidator unit;
	private CreditCardAddressWsDTO creditCard;

	@Before
	public void setUp() {
		MockitoAnnotations.initMocks(this);

		creditCard = new CreditCardAddressWsDTO();

		creditCard.setExpiryMonth(EXPIRY_MONTH);
		creditCard.setExpiryYear(EXPIRY_YEAR);
	}

	@Test
	public void shouldSupportRegistrationFormObject() {
		assertThat(unit.supports(CreditCardAddressWsDTO.class)).isEqualTo(Boolean.TRUE);
	}

	@Test
	public void shouldValidateCreditCardAddressWhenMonthsToExpirationIsNotNegative() {
		doReturn(MONTH_TO_EXPIRATION).when(expiredDateValidatorUtil).getMonthsToExpiration(EXPIRY_MONTH, EXPIRY_YEAR);

		unit.validate(creditCard, errors);

		verify(errors, never()).rejectValue(EXPIRY_MONTH, CREDIT_CARD_EXPIRY_MONTH_INVALID);
		verify(errors, never()).rejectValue(EXPIRY_YEAR, CREDIT_CARD_EXPIRY_YEAR_INVALID);
	}

	@Test
	public void shouldNotValidateCreditCardAddressWhenMonthsToExpirationIsNegative() {
		doReturn(MONTH_TO_EXPIRATION_NEGATIVE).when(expiredDateValidatorUtil)
				.getMonthsToExpiration(EXPIRY_MONTH, EXPIRY_YEAR);

		unit.validate(creditCard, errors);

		verify(errors).rejectValue(EXPIRY_MONTH, CREDIT_CARD_EXPIRY_MONTH_INVALID);
		verify(errors).rejectValue(EXPIRY_YEAR, CREDIT_CARD_EXPIRY_YEAR_INVALID);
	}

}
