package com.paypalocc.validators;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CardTypeData;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;

@UnitTest
public class PayPalCCPaymentInfoValidatorTest {

    private static final String CODE = "code";
    private static final String NAME = "name";
    private static final String PAYPAL = "paypal";
    private static final String PAYPAL_ORDER_ID = "paypalOrderId";
    private static final String CARD_NUMBER = "cardNumber";
    private static final String STATUS = "status";
    private static final String ZERO = "0";
    private static final String PAYER_EMAIL = "payer@e.mail";
    private static final String PAYMENT_DETAILS = "paymentDetails";
    private static final String ACCOUNT_HOLDER_NAME = "accholdername testsurname";
    private static final String NOT_PAY_PAL = "NotPayPal";
    private static final String SEPTEMBER = "9";
    private static final String YEAR = "2024";
    private static final String FEBRUARY = "2";
    private static final String START_MONTH = "startMonth";
    private static final String PAYMENT_START_DATE_INVALID = "payment.startDate.invalid";
    private final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
    private final CardTypeData cardTypeData = new CardTypeData();
    private Errors errors;

    @Mock
    private Validator paymentAddressValidator;

    @InjectMocks
    @Spy
    private PayPalCCPaymentInfoValidator unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        errors = new BeanPropertyBindingResult(paymentInfoData, PAYMENT_DETAILS);

        cardTypeData.setCode(CODE);
        cardTypeData.setName(NAME);
        paymentInfoData.setDefaultPaymentInfo(true);
        paymentInfoData.setCardTypeData(cardTypeData);
        paymentInfoData.setCardType(PAYPAL);
        paymentInfoData.setPayPalOrderId(PAYPAL_ORDER_ID);
        paymentInfoData.setCardNumber(CARD_NUMBER);
        paymentInfoData.setExpirationStatus(STATUS);
        paymentInfoData.setExpiryMonth(ZERO);
        paymentInfoData.setExpiryYear(ZERO);
    }

    @Test
    public void shouldSupportCCPaymentInfoDataClassOnly() {
        assertTrue(unit.supports(CCPaymentInfoData.class));
        assertFalse(unit.supports(String.class));
        assertFalse(unit.supports(GenericBuilder.class));
    }

    @Test
    public void shouldPassForPayPalProviderWhenAccountHolderNameIsNull() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL);
        paymentInfoData.setPayerEmail(PAYER_EMAIL);

        unit.validate(paymentInfoData, errors);

        assertFalse(errors.hasErrors());

        verify(paymentAddressValidator, never()).validate(any(), any());
    }

    @Test
    public void shouldRejectForPayPalProviderWhenPayPalPayerEmailIsNull() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL);

        unit.validate(paymentInfoData, errors);

        assertTrue(errors.hasErrors());
    }

    @Test
    public void shouldPassForNotPayPalProviderWhenAccountHolderNameIsPresent() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        paymentInfoData.setAccountHolderName(ACCOUNT_HOLDER_NAME);
        paymentInfoData.setCardType(NOT_PAY_PAL);

        unit.validate(paymentInfoData, errors);

        assertFalse(errors.hasErrors());
    }

    @Test
    public void shouldRejectForNotPayPalProviderWhenAccountHolderNameIsNull() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        paymentInfoData.setCardType(NOT_PAY_PAL);

        unit.validate(paymentInfoData, errors);

        assertTrue(errors.hasErrors());
    }

    @Test
    public void shouldRejectWhenStartDateIsAfterExpirationDate() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        paymentInfoData.setAccountHolderName(ACCOUNT_HOLDER_NAME);
        paymentInfoData.setCardType(NOT_PAY_PAL);

        paymentInfoData.setStartMonth(SEPTEMBER);
        paymentInfoData.setStartYear(YEAR);
        paymentInfoData.setExpiryMonth(FEBRUARY);
        paymentInfoData.setExpiryYear(YEAR);

        unit.validate(paymentInfoData, errors);

        assertTrue(errors.hasErrors());
        assertEquals(START_MONTH, errors.getFieldError().getField());
        assertEquals(SEPTEMBER, errors.getFieldError().getRejectedValue());
        assertEquals(PAYMENT_START_DATE_INVALID, errors.getFieldError().getCode());
    }

    @Test
    public void shouldPassWhenStartDateIsBeforeExpirationDate() {
        paymentInfoData.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        paymentInfoData.setAccountHolderName(ACCOUNT_HOLDER_NAME);
        paymentInfoData.setCardType(NOT_PAY_PAL);

        paymentInfoData.setStartMonth(FEBRUARY);
        paymentInfoData.setStartYear(YEAR);
        paymentInfoData.setExpiryMonth(SEPTEMBER);
        paymentInfoData.setExpiryYear(YEAR);

        unit.validate(paymentInfoData, errors);

        assertFalse(errors.hasErrors());
    }

}
