package com.paypalocc.controllers;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.data.ws.PayPalCCSetupTokenWsDTO;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import com.paypalocc.populators.PaymentTokenDataPopulator;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commerceservices.request.mapping.annotation.RequestMappingOverride;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsWsDTO;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.HttpClientErrorException;

import javax.annotation.Resource;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}")
public class PayPalPaymentDetailsController extends PayPalBaseController {

    private static final String OBJECT_NAME_PAYMENT_DETAILS = "paymentDetails";
    private static final String FIELDS_TO_MAP = """ 
                    accountHolderName,cardNumber,cardType,issueNumber,startMonth,expiryMonth,startYear,expiryYear,
                    subscriptionId,defaultPaymentInfo,saved,payerEmail,billingAddress(firstName,lastName,titleCode,
                    line1,line2,town,postalCode,region(isocode),country(isocode),defaultAddress)""";

    @Resource
    private PayPalCreditCardFacade payPalCreditCardFacade;
    @Resource
    private PayPalPaymentInfoService payPalPaymentInfoService;
    @Resource
    private PaymentTokenDataPopulator paymentTokenDataPopulator;
    @Resource
    private Validator ccPaymentInfoValidator;

    @RequestMapping(value = "/vault/paypal/setup-token", method = RequestMethod.POST)
    @ResponseStatus(HttpStatus.OK)
    @ApiBaseSiteIdAndUserIdParam
    public PayPalCCSetupTokenWsDTO createVaultSetupToken() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL);

        CCSetupTokenData ccSetupTokenData = payPalCreditCardFacade.requestSetupToken(setupTokenRequestData);
        return getDataMapper().map(ccSetupTokenData, PayPalCCSetupTokenWsDTO.class);
    }

    @RequestMapping(value = "/vault/paypal/payment-token", method = RequestMethod.POST)
    @ApiBaseSiteIdAndUserIdParam
    @ExceptionHandler({HttpClientErrorException.class})
    public ResponseEntity<String> createVaultPaymentToken(@RequestBody final PayPalCCSetupTokenWsDTO setupTokenWsDTO) throws Exception {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenDataPopulator.populate(setupTokenWsDTO, paymentTokenData);
        payPalCreditCardFacade.requestPaymentToken(paymentTokenData);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @DeleteMapping
    @RequestMapping(value = "/vault/paypal")
    @ResponseStatus(value = HttpStatus.OK)
    @ApiBaseSiteIdAndUserIdParam
    public ResponseEntity<String> removePaymentInfoByPk(@RequestBody final String paymentInfoPk) {
        payPalPaymentInfoService.removePaymentInfoByPK(paymentInfoPk);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @RequestMapping(value = "/paymentdetails/{paymentDetailsId}", method = RequestMethod.PATCH, consumes = {MediaType.APPLICATION_JSON_VALUE,
            MediaType.APPLICATION_XML_VALUE})
    @ApiBaseSiteIdAndUserIdParam
    @ResponseStatus(HttpStatus.OK)
    @RequestMappingOverride
    public ResponseEntity<Void> updatePaymentDetails(
            @PathVariable final String paymentDetailsId,
            @RequestBody final PaymentDetailsWsDTO paymentDetails) {
        CCPaymentInfoData paymentInfoData = getUserFacade().getCCPaymentInfoForCode(paymentDetailsId);
        if (paymentInfoData != null) {
            final boolean isAlreadyDefaultPaymentInfo = paymentInfoData.isDefaultPaymentInfo();

            getDataMapper().map(paymentDetails, paymentInfoData, FIELDS_TO_MAP, false);

            validate(paymentInfoData, OBJECT_NAME_PAYMENT_DETAILS, ccPaymentInfoValidator);

            getUserFacade().updateCCPaymentInfo(paymentInfoData);
            if (paymentInfoData.isSaved() && !isAlreadyDefaultPaymentInfo && paymentInfoData.isDefaultPaymentInfo()) {
                getUserFacade().setDefaultPaymentInfo(paymentInfoData);
            }
            return new ResponseEntity<>(HttpStatus.OK);
        }
        return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
    }

}
