/**
 *
 */
package com.paypalocc.controllers;

import com.paypal.hybris.core.exception.ApplePayInvalidAddressException;
import com.paypal.hybris.core.exception.InvalidPaymentInfoException;
import com.paypal.hybris.core.exception.PayPalInvalidAddressException;
import com.paypal.hybris.core.exception.PayPalThreeDSecureVerificationException;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.ApplePayAddressDetailsData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalCreateOrderData;
import com.paypal.hybris.data.PayPalHostedFieldsCreateOrderData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.ws.ApplePayErrorWsDTO;
import com.paypal.hybris.data.ws.ApplePayLineItemWsDto;
import com.paypal.hybris.data.ws.PayPalCreateOrderWsData;
import com.paypal.hybris.data.ws.PayPalHostedFieldsCreateOrderWSData;
import com.paypal.hybris.data.ws.PayPalOrderWsDTO;
import com.paypal.hybris.data.ws.PayPalPaymentRequestWsDTO;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalGuestCheckoutFacade;
import com.paypalocc.populators.ApplePayWsDataPopulator;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsWsDTO;
import de.hybris.platform.commercewebservicescommons.dto.user.AddressWsDTO;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.dto.error.ErrorListWsDTO;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdUserIdAndCartIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import com.paypal.hybris.data.ws.ApplePayAddressDetailsWsData;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import de.hybris.platform.converters.Populator;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import com.paypal.hybris.core.service.PayPalConfigurationService;

import javax.annotation.Resource;

import java.util.List;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.APPLE_PAY_INVALID_SHIPPING_CONTACT_ERROR_CODE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_HOSTED_FIELDS;
import static com.paypalocc.constants.PaypaloccWebConstants.DEFAULT_PAYMENT_METHOD;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}/carts/{cartId}")
@CacheControl(directive = CacheControlDirective.NO_CACHE)
@Api(tags = "Braintree Payment Details")
public class PayPalCheckoutController extends PayPalBaseController {

    private static final Logger LOG = Logger.getLogger(PayPalCheckoutController.class);

    private static final String EXPRESS_CHECKOUT = "EXPRESS_CHECKOUT";
    private static final String MARK_CHECKOUT = "MARK_CHECKOUT";
    private static final String APPLE_PAY_LINE_ITEM_FIELDS = "amount,type,label";

    @Resource(name = "defaultPayPalAcceleratorCheckoutFacade")
    private PayPalAcceleratorCheckoutFacade acceleratorCheckoutFacade;

    @Resource(name = "customerFacade")
    private CustomerFacade customerFacade;

    @Resource(name = "sessionService")
    private SessionService sessionService;

    @Resource(name = "payPalPaymentRequestValidator")
    private Validator payPalPaymentRequestValidator;

    @Resource(name = "payPalGuestCheckoutFacade")
    private PayPalGuestCheckoutFacade payPalGuestCheckoutFacade;

    @Resource(name = "payPalConfigurationService")
    private PayPalConfigurationService payPalConfigurationService;

    @Resource
    private Populator<ApplePayAddressDetailsData, PayPalAddressDetailsData> applePayPayPalAddressDataPopulator;

    @Resource
    private Populator<ApplePayAddressDetailsData, AddressData> applePayAddressDataPopulator;

    @Resource
    private ApplePayWsDataPopulator applePayWsDataPopulator;

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createOrder")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "createPayPalOrder", value = "Create order on PayPal side", notes = "Create and save new PayPal oder")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PayPalOrderWsDTO createPayPalOrder(@RequestBody PayPalCreateOrderWsData createOrderRequest) {
        String orderID;
        ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
        Optional<ApplePayAddressDetailsWsData> applePayAddressDetailsWsData = Optional.ofNullable(createOrderRequest.getApplePayShippingAddress());
        applePayAddressDetailsWsData.ifPresent(address -> applePayWsDataPopulator.populate(address, applePayAddressDetailsData));

        if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(createOrderRequest.getFundingSource())) {
            AddressData addressData = new AddressData();
            applePayAddressDataPopulator.populate(applePayAddressDetailsData, addressData);
            if (!acceleratorCheckoutFacade.isPayPalAddressValid(addressData)) {
                throw new ApplePayInvalidAddressException();
            }
        }
        if (MARK_CHECKOUT.equals(createOrderRequest.getFlow())) {
            Optional<PayPalHostedFieldsCreateOrderWSData> payPalHostedFieldsData = Optional.ofNullable(createOrderRequest.getPayPalHostedFieldsData());
            boolean isShouldBeSaved = false;
            PayPalHostedFieldsCreateOrderData hostedFieldsCreateOrderData = null;
            PayPalCreateOrderData payPalCreateOrderData = new PayPalCreateOrderData();
            payPalCreateOrderData.setFunding(createOrderRequest.getFundingSource());

            if (payPalHostedFieldsData.isPresent()) {
                PayPalHostedFieldsCreateOrderWSData hostedFieldsCreateOrderWSData = payPalHostedFieldsData.get();
                isShouldBeSaved = hostedFieldsCreateOrderWSData.getIsShouldBeSaved();
                hostedFieldsCreateOrderData = GenericBuilder.of(PayPalHostedFieldsCreateOrderData::new)
                        .with(PayPalHostedFieldsCreateOrderData::setBillingAddress,
                                getPayPalAddressDetails(hostedFieldsCreateOrderWSData.getBillingAddress()))
                        .with(PayPalHostedFieldsCreateOrderData::setIsShouldBeSaved, isShouldBeSaved)
                        .with(PayPalHostedFieldsCreateOrderData::setNameOnCard, hostedFieldsCreateOrderWSData.getNameOnCard())
                        .build();
                payPalCreateOrderData.setHostedFieldsData(hostedFieldsCreateOrderData);
            } else if (applePayAddressDetailsWsData.isPresent()) {
                payPalCreateOrderData.setApplePayShippingAddress(applePayAddressDetailsData);
            } else {
                isShouldBeSaved = createOrderRequest.isSavePaymentMethod();
                payPalCreateOrderData.setSavePaymentMethod(isShouldBeSaved);
            }
            orderID = acceleratorCheckoutFacade.createPayPalOrder(payPalCreateOrderData);
            CreditCardType paymentType = acceleratorCheckoutFacade.setCreditCardType(createOrderRequest.getFundingSource());
            String nameOnCard = hostedFieldsCreateOrderData != null ? hostedFieldsCreateOrderData.getNameOnCard() : StringUtils.EMPTY;
            acceleratorCheckoutFacade.createPayPalPaymentSubscriptionForMarkCheckout(orderID, paymentType, nameOnCard, isShouldBeSaved);
        } else {
            if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(createOrderRequest.getFundingSource())) {
                orderID = acceleratorCheckoutFacade.createApplePayOrderForExpressCheckout(applePayAddressDetailsData);
            } else {
                orderID = acceleratorCheckoutFacade.createPayPalOrderForExpressCheckout();
            }
        }
        return GenericBuilder.of(PayPalOrderWsDTO::new)
                .with(PayPalOrderWsDTO::setOrderId, orderID).build();
    }


    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/createPayPalPayment")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public PaymentDetailsWsDTO createPayPalPaymentMethod(
            @ApiParam(value = "Request body parameter that contains" +
                    "payPalOrderId and payPAlFlow option ", required = true) @RequestBody final PayPalPaymentRequestWsDTO payPalRequest,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
                    allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
            @ApiParam(value = "Payment method type") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_PAYMENT_METHOD) final String paymentMethodType) {
        validate(payPalRequest, "payPalRequest", payPalPaymentRequestValidator);
        final String checkoutType = payPalRequest.getPayPalFlow();
        final String fundingSource = payPalRequest.getFundingSource();
        final PayPalOrderDetailsData orderDetailsData = acceleratorCheckoutFacade.getPayPalOrderDetails();
        if (CreditCardType.PAYPAL_HOSTED_FIELDS_CARD.name().equalsIgnoreCase(fundingSource)) {
            orderDetailsData.setBillingAddress(getPayPalAddressDetails(payPalRequest.getBillingAddress()));
        }
        if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(fundingSource)) {
            PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
            ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
            applePayWsDataPopulator.populate(payPalRequest.getApplePayBillingAddress(), applePayAddressDetailsData);
            applePayPayPalAddressDataPopulator.populate(applePayAddressDetailsData, payPalAddressDetailsData);
            payPalAddressDetailsData.setEmail(payPalRequest.getEmail());
            orderDetailsData.setBillingAddress(payPalAddressDetailsData);
            orderDetailsData.getShippingAddress().setEmail(payPalRequest.getEmail());
        }
        try {
            if (StringUtils.equals(EXPRESS_CHECKOUT, checkoutType)) {
                if (getUserFacade().isAnonymousUser()) {
                    payPalGuestCheckoutFacade.processCheckoutForAnonymous(orderDetailsData.getShippingAddress());
                }
                acceleratorCheckoutFacade.processExpressCheckout(orderDetailsData, paymentMethodType);
            } else if (StringUtils.equals(MARK_CHECKOUT, checkoutType)) {
                if (PAYPAL_HOSTED_FIELDS.equals(fundingSource) && 
                        (isHostedFields3DSVerificationRequiredButNotProvided(orderDetailsData) 
                                || !acceleratorCheckoutFacade.isThreeDsVerificationSuccess(orderDetailsData.getAuthenticationResult()))) {
                    throw new PayPalThreeDSecureVerificationException();
                }

                final String orderId = acceleratorCheckoutFacade.getCheckoutCart().getPayPalOrderId();
                final CCPaymentInfoData paymentInfo = acceleratorCheckoutFacade.getCCPaymentInfoByOrderId(orderId);

                CCPaymentInfoData payPalPaymentSubscription = acceleratorCheckoutFacade
                        .updatePayPalPaymentSubscription(orderDetailsData, paymentInfo);
                acceleratorCheckoutFacade.setPaymentDetails(payPalPaymentSubscription.getId());
            }
        } catch (ConversionException e) {
            LOG.warn("This country not support delivery mode", e);
            throw new PayPalInvalidAddressException("No country with the code");
        }

        return getDataMapper()
                .map(acceleratorCheckoutFacade.getCheckoutCart().getPaymentInfo(), PaymentDetailsWsDTO.class, fields);
    }

    private boolean isHostedFields3DSVerificationRequiredButNotProvided(PayPalOrderDetailsData orderDetailsData) {
        return payPalConfigurationService.isThreeDsVerificationOnCheckoutAlwaysRequiredEnable() &&
                orderDetailsData.getAuthenticationResult() == null;
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/paypal/paymentInfo/payPalPayment")
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    @ResponseStatus(HttpStatus.OK)
    public PaymentDetailsWsDTO processExpressCheckout() {
        return null;
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @RequestMapping(value = "/paypal/paymentdetails", method = RequestMethod.PUT)
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "replaceCartPaymentDetails", value = "Sets credit card payment details for the cart.", notes = "Sets credit card payment details for the cart.")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public void replaceCartPaymentDetails(
            @ApiParam(value = "Payment details identifier.", required = true) @RequestParam final String paymentDetailsId)
            throws InvalidPaymentInfoException {
        setSelectedPaymentInfoToCartInternal(paymentDetailsId);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_ANONYMOUS", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/apple-pay/shipping-address/update")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(nickname = "updateShippingAddressFromApplePay", value = "Updates shipping details for the cart.", notes = "Updates shipping details for the cart.")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public List<ApplePayLineItemWsDto> updateShippingAddressFromApplePay(@RequestBody ApplePayAddressDetailsWsData shippingAddress) {
        ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
        applePayWsDataPopulator.populate(shippingAddress, applePayAddressDetailsData);

        acceleratorCheckoutFacade.updateApplePayShippingAddress(applePayAddressDetailsData);

        return getDataMapper().mapAsList(acceleratorCheckoutFacade.getApplePayLineItemsFromCart(), ApplePayLineItemWsDto.class, APPLE_PAY_LINE_ITEM_FIELDS);
    }

    @ExceptionHandler(ApplePayInvalidAddressException.class)
    @ResponseStatus(HttpStatus.UNPROCESSABLE_ENTITY)
    public ApplePayErrorWsDTO handleApplePayInvalidAddressException(ApplePayInvalidAddressException e){
        return GenericBuilder.of(ApplePayErrorWsDTO::new)
                .with(ApplePayErrorWsDTO::setCode, APPLE_PAY_INVALID_SHIPPING_CONTACT_ERROR_CODE)
                .with(ApplePayErrorWsDTO::setContactField, e.getInvalidField())
                .build();
    }
    private void setSelectedPaymentInfoToCartInternal(String paymentDetailsId) throws InvalidPaymentInfoException {
        if (!acceleratorCheckoutFacade.setSelectedPaymentInfoToCart(paymentDetailsId)) {
            throw new InvalidPaymentInfoException(paymentDetailsId);
        }
    }

    private PayPalAddressDetailsData getPayPalAddressDetails(AddressWsDTO addressWsDTO) {
        return GenericBuilder.of(PayPalAddressDetailsData::new)
                .with(PayPalAddressDetailsData::setFirstName, addressWsDTO.getFirstName())
                .with(PayPalAddressDetailsData::setLastName, addressWsDTO.getLastName())
                .with(PayPalAddressDetailsData::setLine1, addressWsDTO.getLine1())
                .with(PayPalAddressDetailsData::setLine2, addressWsDTO.getLine2())
                .with(PayPalAddressDetailsData::setCity, addressWsDTO.getTown())
                .with(PayPalAddressDetailsData::setRegion, addressWsDTO.getRegion() == null ? null : addressWsDTO.getRegion().getIsocodeShort())
                .with(PayPalAddressDetailsData::setPostalCode, addressWsDTO.getPostalCode())
                .with(PayPalAddressDetailsData::setCountryCode, addressWsDTO.getCountry().getIsocode())
                .build();
    }

}
