package com.paypal.hybris.b2baddon.b2bfacade;

import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.model.PaypalAccessTokenModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.core.service.PaypalAccessTokenService;
import com.paypal.hybris.data.PayPalConnectB2BRegisterData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.data.PayPalUserExistB2BData;
import com.paypal.hybris.facade.exception.PaypalAccessTokenNotFoundException;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.b2b.model.B2BCustomerModel;
import de.hybris.platform.b2b.model.B2BRegistrationModel;
import de.hybris.platform.cms2.model.site.CMSSiteModel;
import de.hybris.platform.cms2.servicelayer.services.CMSSiteService;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.c2l.LanguageModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.EmployeeModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import de.hybris.platform.tx.Transaction;
import de.hybris.platform.workflow.WorkflowProcessingService;
import de.hybris.platform.workflow.WorkflowService;
import de.hybris.platform.workflow.WorkflowTemplateService;
import de.hybris.platform.workflow.model.WorkflowModel;
import de.hybris.platform.workflow.model.WorkflowTemplateModel;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Optional;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalB2BRegistrationUserFacadeTest {

    private static final String ACCESS_TOKEN = "accessToken";
    private static final String NAME = "Name";
    private static final String PAYER_ID = "payerId";
    private static final String EMAIL = "some@email.com";
    private static final String USER_UID = "userUid";
    private static final String TITLE_CODE = "titleCode";
    private static final String US = "US";
    @Spy
    @InjectMocks
    private DefaultPayPalB2BRegistrationUserFacade unit;
    @Mock
    private PayPalRegistrationUserFacade registrationUserFacade;
    @Mock
    private PayPalConnectService payPalConnectService;
    @Mock
    private ModelService modelService;
    @Mock
    private UserService userService;
    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private WorkflowTemplateService workflowTemplateService;
    @Mock
    private WorkflowService workflowService;
    @Mock
    private WorkflowProcessingService workflowProcessingService;
    @Mock
    private BaseStoreService baseStoreService;
    @Mock
    private CMSSiteService cmsSiteService;
    @Mock
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    @Mock
    private PaypalAccessTokenService paypalAccessTokenService;
    @Mock
    private PayPalConfigurationService payPalConfigurationService;

    @Mock
    private PaypalAccessTokenModel accessTokenModel;
    @Mock
    private B2BCustomerModel b2BCustomerModel;
    @Mock
    private UserModel userModel;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private EmployeeModel employeeModel;
    @Mock
    private Transaction transaction;
    @Mock
    private B2BRegistrationModel b2BRegistrationModel;
    @Mock
    private WorkflowTemplateModel workflowTemplateModel;
    @Mock
    private WorkflowModel workflowModel;
    @Mock
    private CurrencyModel currencyModel;
    @Mock
    private LanguageModel languageModel;
    @Mock
    private CountryModel countryModel;
    @Mock
    private TitleModel titleModel;
    @Mock
    private BaseStoreModel baseStoreModel;
    @Mock
    private CMSSiteModel cmsSiteModel;

    private PayPalConnectB2BRegisterData payPalConnectB2BRegisterData;
    private PayPalConnectUserData payPalConnectUserData;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        payPalConnectUserData = new PayPalConnectUserData();
        payPalConnectB2BRegisterData = new PayPalConnectB2BRegisterData();

        payPalConnectUserData.setName(NAME);
        payPalConnectUserData.setPayer_id(PAYER_ID);

        payPalConnectB2BRegisterData.setAccessTokenGuid(ACCESS_TOKEN);
        payPalConnectB2BRegisterData.setName(NAME);
        payPalConnectB2BRegisterData.setEmail(EMAIL);
        payPalConnectB2BRegisterData.setTitleCode(TITLE_CODE);
        payPalConnectB2BRegisterData.setCompanyAddressCountryIso(US);

        when(accessTokenModel.getAccessToken()).thenReturn(ACCESS_TOKEN);

        when(registrationUserFacade.getEmailFromPayPalUserData(payPalConnectUserData)).thenReturn(EMAIL);
        when(payPalConnectService.getUserDataByAccessToken(ACCESS_TOKEN)).thenReturn(payPalConnectUserData);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(cmsSiteService.getCurrentSite()).thenReturn(cmsSiteModel);
    }

    @Test(expected = PaypalAccessTokenNotFoundException.class)
    public void shouldThrowPaypalAccessTokenNotFoundExceptionWhenAccessTokenIsNull() throws PaypalAccessTokenNotFoundException {
        when(paypalAccessTokenService.getPaypalAccessToken(null)).thenReturn(Optional.empty());

        unit.registerB2BPayPalUser(payPalConnectB2BRegisterData);
    }

    @Test
    public void shouldReturnEmptyPayerIdWhenUserAlreadyExists() throws PaypalAccessTokenNotFoundException {
        when(paypalAccessTokenService.getPaypalAccessToken(any())).thenReturn(Optional.of(accessTokenModel));
        when(userService.isUserExisting(EMAIL)).thenReturn(true);

        try (MockedStatic<Transaction> transactionMockedStatic = mockStatic(Transaction.class)) {
            transactionMockedStatic.when(Transaction::current).thenReturn(transaction);

            String payerIdResult = unit.registerB2BPayPalUser(payPalConnectB2BRegisterData);

            verify(transaction).rollback();
            assertEquals(StringUtils.EMPTY, payerIdResult);
        }

    }

    @Test
    public void shouldRegister() throws PaypalAccessTokenNotFoundException {
        when(paypalAccessTokenService.getPaypalAccessToken(any())).thenReturn(Optional.of(accessTokenModel));
        when(userService.isUserExisting(EMAIL)).thenReturn(false);
        doReturn(b2BRegistrationModel).when(unit).toRegistrationModel(payPalConnectB2BRegisterData);
        doReturn(b2BCustomerModel).when(unit).toCustomerModel(payPalConnectB2BRegisterData);
        when(workflowTemplateService.getWorkflowTemplateForCode(any())).thenReturn(workflowTemplateModel);
        when(userService.getAdminUser()).thenReturn(employeeModel);
        when(workflowService.createWorkflow(workflowTemplateModel, b2BRegistrationModel, employeeModel)).thenReturn(workflowModel);
        when(payPalConfigurationService.isPayPalConnectAddPaymentMethodFlow()).thenReturn(false);
        doNothing().when(paypalAccessTokenService).removePaypalAccessToken(ACCESS_TOKEN);
        when(workflowProcessingService.startWorkflow(workflowModel)).thenReturn(true);
        when(registrationUserFacade.getEmailFromPayPalUserData(payPalConnectUserData)).thenReturn(EMAIL);
        doNothing().when(registrationUserFacade).setPayerIdToUser(EMAIL, PAYER_ID);

        try (MockedStatic<Transaction> transactionMockedStatic = mockStatic(Transaction.class)) {
            transactionMockedStatic.when(Transaction::current).thenReturn(transaction);

            String payerIdResult = unit.registerB2BPayPalUser(payPalConnectB2BRegisterData);

            verify(modelService).save(b2BRegistrationModel);
            verify(modelService).save(b2BCustomerModel);
            verify(modelService).save(workflowModel);
            verify(paypalAccessTokenService).removePaypalAccessToken(ACCESS_TOKEN);
            verify(workflowProcessingService).startWorkflow(workflowModel);
            verify(transaction).commit();
            verify(registrationUserFacade).setPayerIdToUser(EMAIL, PAYER_ID);

            assertEquals(PAYER_ID, payerIdResult);
        }

    }

    @Test
    public void shouldReturnCustomerExistsAndApprovedWhenB2BCustomerExistsByUid() {
        when(userService.getUserForUID(USER_UID)).thenReturn(b2BCustomerModel);

        PayPalUserExistB2BData existB2BData = unit.isCustomerExistAndApprovedByUID(USER_UID);
        assertEquals(true, existB2BData.getIsExist());
        assertEquals(true, existB2BData.getIsApproved());
    }

    @Test
    public void shouldReturnCustomerNotExistsAndNotApprovedWhenB2BCustomerDoesNotExistByUid() {
        when(userService.getUserForUID(USER_UID)).thenThrow(UnknownIdentifierException.class);

        PayPalUserExistB2BData existB2BData = unit.isCustomerExistAndApprovedByUID(USER_UID);
        assertEquals(false, existB2BData.getIsExist());
        assertEquals(false, existB2BData.getIsApproved());
    }

    @Test
    public void shouldReturnCustomerExistsAndNotApprovedWhenCustomerExistByUid() {
        when(userService.getUserForUID(USER_UID)).thenReturn(userModel);

        PayPalUserExistB2BData existB2BData = unit.isCustomerExistAndApprovedByUID(USER_UID);
        assertEquals(true, existB2BData.getIsExist());
        assertEquals(false, existB2BData.getIsApproved());
    }

    @Test
    public void shouldReturnCustomerExistsAndApprovedWhenB2BCustomerExistsByPayerId() {
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(USER_UID)).thenReturn(b2BCustomerModel);

        PayPalUserExistB2BData existB2BData = unit.isCustomerExistAndApprovedByPayerId(USER_UID);
        assertEquals(true, existB2BData.getIsExist());
        assertEquals(true, existB2BData.getIsApproved());
    }

    @Test
    public void shouldReturnCustomerNotExistsAndNotApprovedWhenB2BCustomerDoesNotExistByPayerId() {
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(USER_UID)).thenReturn(null);

        PayPalUserExistB2BData existB2BData = unit.isCustomerExistAndApprovedByPayerId(USER_UID);
        assertEquals(false, existB2BData.getIsExist());
        assertEquals(false, existB2BData.getIsApproved());
    }

    @Test
    public void shouldReturnCustomerModel() {
        when(modelService.create(CustomerModel.class)).thenReturn(customerModel);
        when(commonI18NService.getCurrentCurrency()).thenReturn(currencyModel);
        when(commonI18NService.getCurrentLanguage()).thenReturn(languageModel);
        when(userService.getTitleForCode(TITLE_CODE)).thenReturn(titleModel);

        CustomerModel customerModel = unit.toCustomerModel(payPalConnectB2BRegisterData);

        verify(customerModel).setName(StringUtils.capitalize(NAME));
        verify(customerModel).setLoginDisabled(true);
        verify(customerModel).setSessionCurrency(currencyModel);
        verify(customerModel).setSessionLanguage(languageModel);
        verify(customerModel).setPasswordForSingleAuthenticationActive(false);
        verify(customerModel).setTitle(titleModel);
    }

    @Test
    public void shouldReturnRegistrationModel() {
        when(modelService.create(B2BRegistrationModel.class)).thenReturn(b2BRegistrationModel);
        when(commonI18NService.getCurrentCurrency()).thenReturn(currencyModel);
        when(commonI18NService.getCurrentLanguage()).thenReturn(languageModel);
        when(commonI18NService.getCountry(US)).thenReturn(countryModel);
        when(userService.getTitleForCode(TITLE_CODE)).thenReturn(titleModel);

        B2BRegistrationModel registrationModel = unit.toRegistrationModel(payPalConnectB2BRegisterData);

        verify(registrationModel).setName(StringUtils.capitalize(NAME));
        verify(registrationModel).setBaseSite(cmsSiteModel);
        verify(registrationModel).setBaseStore(baseStoreModel);
        verify(registrationModel).setCurrency(currencyModel);
        verify(registrationModel).setLanguage(languageModel);
        verify(registrationModel).setTitle(titleModel);
    }

}
