package com.paypal.hybris.b2baddon.forms.validation;

import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorstorefrontcommons.constants.WebConstants;
import de.hybris.platform.b2bacceleratoraddon.forms.RegistrationForm;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.configuration.Configuration;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.validation.Errors;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoInteractions;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalSecurePortalRegistrationValidatorTest {

    private static final String COMPANY_ADDRESS_STREET = "companyAddressStreet";
    private static final String COMPANY_ADDRESS_CITY = "companyAddressCity";
    private static final String COMPANY_NAME = "companyName";
    private static final String COMPANY_ADDRESS_COUNTRY_ISO = "companyAddressCountryIso";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String POSITION = "position";
    private static final String COMPANY_ADDRESS_POSTAL_CODE = "companyAddressPostalCode";
    private static final String TELEPHONE = "telephone";
    private static final String TITLE_CODE = "titleCode";
    private static final String COMPANY_ADDRESS_STREET_LINE_2 = "companyAddressStreetLine2";
    private static final String TELEPHONE_EXTENSION = "telephoneExtension";
    private static final String EMAIL = "email";
    private static final String VALID_EMAIL = "some@email.com";
    private static final String MANDATORY_FIELD_ERROR = "text.secureportal.register.field.mandatory";
    private static final String TOO_LONG_FIELD_ERROR = "text.secureportal.register.field.toolong";
    private static final String INVALID_EMAIL_FIELD_ERROR = "text.secureportal.register.email.invalid";
    private static final String PAYER_ID_PRESENT_FIELD_ERROR = "paypal.connect.registration.payerId.global.error";
    private static final String EMAIL_REGEX = "\\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]+\\b";
    private static final String TOO_LONG_STRING = new String(new byte[256]);

    @InjectMocks
    private PayPalSecurePortalRegistrationValidator unit;

    @Mock
    private UserService userService;
    @Mock
    private PayPalCustomerFacade payPalCustomerFacade;
    @Mock
    private ConfigurationService configurationService;
    @Mock
    private Configuration configuration;
    @Mock
    private Errors errors;
    private RegistrationForm registrationForm;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        registrationForm = new RegistrationForm();
        when(userService.isUserExisting(EMAIL)).thenReturn(false);
        when(configurationService.getConfiguration()).thenReturn(configuration);
        when(configuration.getString(WebConstants.EMAIL_REGEX)).thenReturn(EMAIL_REGEX);
    }

    @Test
    public void shouldValidateValidRegistrationForm() {
        registrationForm.setCompanyAddressStreet(COMPANY_ADDRESS_STREET);
        registrationForm.setCompanyAddressCity(COMPANY_ADDRESS_CITY);
        registrationForm.setCompanyName(COMPANY_NAME);
        registrationForm.setCompanyAddressCountryIso(COMPANY_ADDRESS_COUNTRY_ISO);
        registrationForm.setPosition(POSITION);
        registrationForm.setCompanyAddressPostalCode(COMPANY_ADDRESS_POSTAL_CODE);
        registrationForm.setTelephone(TELEPHONE);

        registrationForm.setCompanyAddressStreetLine2(COMPANY_ADDRESS_STREET_LINE_2);
        registrationForm.setEmail(VALID_EMAIL);
        registrationForm.setFirstName(FIRST_NAME);
        registrationForm.setLastName(LAST_NAME);
        registrationForm.setTelephoneExtension(TELEPHONE_EXTENSION);
        registrationForm.setTitleCode(TITLE_CODE);

        unit.validate(registrationForm, errors);

        verifyNoInteractions(errors);
    }

    @Test
    public void shouldValidateEmptyForm() {
        unit.validate(registrationForm, errors);

        verify(errors).rejectValue(COMPANY_ADDRESS_STREET, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(COMPANY_ADDRESS_CITY, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(COMPANY_NAME, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(COMPANY_ADDRESS_COUNTRY_ISO, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(FIRST_NAME, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(LAST_NAME, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(POSITION, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(COMPANY_ADDRESS_POSTAL_CODE, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(TELEPHONE, MANDATORY_FIELD_ERROR);
        verify(errors).rejectValue(EMAIL, MANDATORY_FIELD_ERROR);

        verify(errors, times(10)).rejectValue(any(), eq(MANDATORY_FIELD_ERROR));
        verify(errors, never()).rejectValue(any(), eq(INVALID_EMAIL_FIELD_ERROR));
        verify(errors, never()).rejectValue(any(), eq(TOO_LONG_FIELD_ERROR));
    }

    @Test
    public void shouldValidateOptionalFieldsLengths() {
        registrationForm.setTitleCode(TOO_LONG_STRING);
        registrationForm.setCompanyAddressStreetLine2(TOO_LONG_STRING);
        registrationForm.setTelephoneExtension(TOO_LONG_STRING);

        unit.validate(registrationForm, errors);

        verify(errors, times(10)).rejectValue(any(), eq(MANDATORY_FIELD_ERROR));
        verify(errors, times(3)).rejectValue(any(), eq(TOO_LONG_FIELD_ERROR));
        verify(errors, never()).rejectValue(any(), eq(INVALID_EMAIL_FIELD_ERROR));
    }

    @Test
    public void shouldValidateEmailField() {
        registrationForm.setEmail(EMAIL);

        unit.validate(registrationForm, errors);

        verify(errors, times(9)).rejectValue(any(), eq(MANDATORY_FIELD_ERROR));
        verify(errors, times(1)).rejectValue(any(), eq(INVALID_EMAIL_FIELD_ERROR));
        verify(errors, never()).rejectValue(any(), eq(TOO_LONG_FIELD_ERROR));
    }

    @Test
    public void shouldValidateExistingUserWithPayerId() {
        registrationForm.setEmail(EMAIL);
        when(userService.isUserExisting(EMAIL)).thenReturn(true);
        when(payPalCustomerFacade.isPayerIdInCustomer(EMAIL)).thenReturn(true);

        unit.validate(registrationForm, errors);

        verify(errors, times(9)).rejectValue(any(), eq(MANDATORY_FIELD_ERROR));
        verify(errors, never()).rejectValue(any(), eq(INVALID_EMAIL_FIELD_ERROR));
        verify(errors, times(1)).rejectValue(any(), eq(PAYER_ID_PRESENT_FIELD_ERROR));
    }

    @Test
    public void shouldSupportRegistrationFormObject() {
        assertTrue(unit.supports(RegistrationForm.class));
    }

    @Test
    public void shouldNotSupportNonRegistrationFormObject() {
        assertFalse(unit.supports(Object.class));
    }

}
