package com.paypal.hybris.b2baddon.controllers.pages;

import com.paypal.hybris.b2baddon.constants.Paypalb2baddonWebConstants;
import com.paypal.hybris.b2baddon.controllers.Paypalb2baddonControllerConstants;
import com.paypal.hybris.core.exception.PayPalPaymentAuthorizationException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.exception.PayPalCreditCardOrderAmountWasChangedException;
import com.paypal.hybris.data.PayPalPlaceOrderFormData;
import com.paypal.hybris.b2bfacade.PayPalB2BCheckoutFacade;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.facade.facades.PayPalSessionPaymentMethodFacade;
import de.hybris.platform.acceleratorservices.enums.CheckoutPciOptionEnum;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.PreValidateCheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.PreValidateQuoteCheckoutStep;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.RequireHardLogIn;
import de.hybris.platform.acceleratorstorefrontcommons.constants.WebConstants;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.b2bacceleratoraddon.controllers.pages.checkout.steps.SummaryCheckoutStepController;
import de.hybris.platform.b2bacceleratoraddon.forms.PlaceOrderForm;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.exception.EntityValidationException;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BReplenishmentRecurrenceEnum;
import de.hybris.platform.b2bacceleratorservices.enums.CheckoutPaymentType;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.order.data.AbstractOrderData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.commercefacades.product.ProductOption;
import de.hybris.platform.commercefacades.product.data.ProductData;
import de.hybris.platform.commerceservices.order.CommerceCartModificationException;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.cronjob.enums.DayOfWeek;
import javax.annotation.Resource;
import org.apache.commons.collections.CollectionUtils;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


@Controller
@RequestMapping(value = "/checkout/multi/summary")
public class PayPalB2BSummaryCheckoutStepController extends SummaryCheckoutStepController {

    private static final Logger LOG = Logger.getLogger(PayPalB2BSummaryCheckoutStepController.class);

    private static final String SUMMARY = "summary";
    private static final String FAILED_DURING_PLACING_ORDER = "Failed to place Order: ";
    private static final String PLACE_ORDER_FORM = "placeOrderForm";
    private static final String DEFAULT_REPLENISHMENT_DAYS_NUMBER = "14";

    @Resource(name = "payPalSessionPaymentMethodFacade")
    private PayPalSessionPaymentMethodFacade payPalSessionPaymentMethodFacade;

    @Resource(name = "defaultB2BAcceleratorCheckoutFacade")
    private PayPalB2BCheckoutFacade payPalB2BCheckoutFacade;

    @PostMapping(value = "/placeOrder")
    @PreValidateQuoteCheckoutStep
    @RequireHardLogIn
    public String placeOrder(@ModelAttribute("placeOrderForm") final PlaceOrderForm placeOrderForm, final Model model,
        final RedirectAttributes redirectModel)
        throws CMSItemNotFoundException, InvalidCartException, CommerceCartModificationException {
        if (validateOrderForm(placeOrderForm, model)) {
            return enterStep(model, redirectModel);
        }

        // authorize, if failure occurs don't allow to place the order
        try {
            validateAndAuthorizePayment(getCartFacade().getSessionCart(),
                GenericBuilder
                    .of(PayPalPlaceOrderFormData::new)
                    .with(PayPalPlaceOrderFormData::setSecurityCode, placeOrderForm.getSecurityCode())
                    .with(PayPalPlaceOrderFormData::setReplenishment, placeOrderForm.isReplenishmentOrder())
                    .build());
        } catch (final AdapterException ae) {
            String message = FAILED_DURING_PLACING_ORDER + ae.getMessage();
            LOG.error(message, ae);
        } catch (PayPalPaymentAuthorizationException e) {
            GlobalMessages.addErrorMessage(model, e.getMessage());
            return enterStep(model, redirectModel);
        } catch (final PayPalProcessPaymentException | PayPalCreditCardOrderAmountWasChangedException ex) {
            String message = FAILED_DURING_PLACING_ORDER + ex.getMessage();
            LOG.error(message, ex);
            GlobalMessages.addFlashMessage(redirectModel, GlobalMessages.ERROR_MESSAGES_HOLDER, ex.getMessage());
            return getCheckoutStep().previousStep();
        }

        final PlaceOrderData placeOrderData = new PlaceOrderData();
        placeOrderData.setNDays(placeOrderForm.getnDays());
        placeOrderData.setNDaysOfWeek(placeOrderForm.getnDaysOfWeek());
        placeOrderData.setNthDayOfMonth(placeOrderForm.getNthDayOfMonth());
        placeOrderData.setNWeeks(placeOrderForm.getnWeeks());
        placeOrderData.setReplenishmentOrder(placeOrderForm.isReplenishmentOrder());
        placeOrderData.setReplenishmentRecurrence(placeOrderForm.getReplenishmentRecurrence());
        placeOrderData.setReplenishmentStartDate(placeOrderForm.getReplenishmentStartDate());
        placeOrderData.setSecurityCode(placeOrderForm.getSecurityCode());
        placeOrderData.setTermsCheck(placeOrderForm.isTermsCheck());

        final AbstractOrderData orderData;
        try {
            orderData = getB2BCheckoutFacade().placeOrder(placeOrderData);
        } catch (final EntityValidationException e) {
            String message = FAILED_DURING_PLACING_ORDER + e.getMessage();
            LOG.error(message, e);
            GlobalMessages.addErrorMessage(model, e.getLocalizedMessage());

            placeOrderForm.setTermsCheck(false);
            model.addAttribute(placeOrderForm);

            return enterStep(model, redirectModel);
        } catch (final Exception e) {
            String message = FAILED_DURING_PLACING_ORDER + e.getMessage();
            LOG.error(message, e);
            GlobalMessages.addErrorMessage(model, "checkout.placeOrder.failed");
            return enterStep(model, redirectModel);
        }

        return redirectToOrderConfirmationPage(placeOrderData, orderData);
    }

    @RequestMapping(value = "/view", method = RequestMethod.GET)
    @RequireHardLogIn
    @Override
    @PreValidateQuoteCheckoutStep
    @PreValidateCheckoutStep(checkoutStep = SUMMARY)
    public String enterStep(final Model model, final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException, CommerceCartModificationException {
        if (payPalSessionPaymentMethodFacade.isSessionPaymentFlowEnable()
                && payPalSessionPaymentMethodFacade.isPaymentMethodTypeEligible()
                && !payPalSessionPaymentMethodFacade.isPaymentMethodValid()) {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    Paypalb2baddonWebConstants.CHECKOUT_SUMMARY_SESSION_PAYMENT_ERROR);
            return REDIRECT_PREFIX + Paypalb2baddonWebConstants.PAYPAL_CHECKOUT_MULTI_ADD_PAYMENT_PAGE;
        }
        final CartData cartData = getCheckoutFacade().getCheckoutCart();
        if (CollectionUtils.isNotEmpty(cartData.getEntries())) {
            for (final OrderEntryData entry : cartData.getEntries()) {
                final String productCode = entry.getProduct().getCode();
                final ProductData product = getProductFacade().getProductForCodeAndOptions(productCode, Arrays.asList(
                        ProductOption.BASIC, ProductOption.PRICE, ProductOption.VARIANT_MATRIX_BASE, ProductOption.PRICE_RANGE));
                entry.setProduct(product);
            }
        }

        model.addAttribute("cartData", cartData);
        model.addAttribute("allItems", cartData.getEntries());
        model.addAttribute("deliveryAddress", cartData.getDeliveryAddress());
        model.addAttribute("deliveryMode", cartData.getDeliveryMode());
        model.addAttribute("paymentInfo", cartData.getPaymentInfo());
        model.addAttribute("nDays", getNumberRange(1, 30));
        model.addAttribute("nthDayOfMonth", getNumberRange(1, 31));
        model.addAttribute("nthWeek", getNumberRange(1, 12));
        model.addAttribute("daysOfWeek", getB2BCheckoutFacade().getDaysOfWeekForReplenishmentCheckoutSummary());

        // Only request the security code if the SubscriptionPciOption is set to Default.
        final boolean requestSecurityCode = (CheckoutPciOptionEnum.DEFAULT
                .equals(getCheckoutFlowFacade().getSubscriptionPciOption()));
        model.addAttribute("requestSecurityCode", Boolean.valueOf(requestSecurityCode));

        if (!model.containsAttribute(PLACE_ORDER_FORM)) {
            final PlaceOrderForm placeOrderForm = new PlaceOrderForm();
            placeOrderForm.setReplenishmentRecurrence(B2BReplenishmentRecurrenceEnum.MONTHLY);
            placeOrderForm.setnDays(DEFAULT_REPLENISHMENT_DAYS_NUMBER);
            final List<DayOfWeek> daysOfWeek = new ArrayList<DayOfWeek>();
            daysOfWeek.add(DayOfWeek.MONDAY);
            placeOrderForm.setnDaysOfWeek(daysOfWeek);
            model.addAttribute(PLACE_ORDER_FORM, placeOrderForm);
        }

        final ContentPageModel multiCheckoutSummaryPage = getContentPageForLabelOrId(MULTI_CHECKOUT_SUMMARY_CMS_PAGE_LABEL);
        storeCmsPageInModel(model, multiCheckoutSummaryPage);
        setUpMetaDataForContentPage(model, multiCheckoutSummaryPage);
        model.addAttribute(WebConstants.BREADCRUMBS_KEY,
                getResourceBreadcrumbBuilder().getBreadcrumbs("checkout.multi.summary.breadcrumb"));
        model.addAttribute("metaRobots", "noindex,nofollow");
        setCheckoutStepLinksForModel(model, getCheckoutStep());
        return Paypalb2baddonControllerConstants.Views.Pages.Checkout.CHECKOUT_SUMMARY_PAGE;
    }

    protected void validateAndAuthorizePayment(final CartData cartData, final PayPalPlaceOrderFormData placeOrderFormData)
            throws PayPalPaymentAuthorizationException {
        if (CheckoutPaymentType.CARD.getCode().equals(cartData.getPaymentType().getCode())
                && !payPalB2BCheckoutFacade.authorizePayment(placeOrderFormData)) {
            throw new PayPalPaymentAuthorizationException("checkout.error.authorization.failed");
        }
    }

}
