package com.paypal.hybris.b2baddon.controllers.pages;


import com.google.gson.Gson;
import com.paypal.enums.PayPalPageType;
import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.b2baddon.constants.Paypalb2baddonWebConstants;
import com.paypal.hybris.addon.data.PayPalPayerIdStatusData;
import com.paypal.hybris.addon.errors.PaypalValidationError;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.addon.forms.CreditCardAddressForm;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.b2baddon.constants.Paypalb2baddonWebConstants;
import com.paypal.hybris.b2baddon.validators.PayPalPasswordValidator;
import com.paypal.hybris.b2baddon.validators.PaypalExpiredDateValidator;
import com.paypal.hybris.b2bfacade.impl.DefaultPayPalB2BCreditCardFacade;
import com.paypal.hybris.core.exception.PayPalCMSItemNotFoundException;
import com.paypal.hybris.core.exception.PayPalCreditCardRemovalException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentInfoService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalConfigurationFacade;
import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCreditCardFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalSetUpPasswordFacade;
import com.paypal.hybris.facade.strategy.configuration.page.PageConfigurationStrategy;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.RequireHardLogIn;
import de.hybris.platform.acceleratorstorefrontcommons.breadcrumb.Breadcrumb;
import de.hybris.platform.acceleratorstorefrontcommons.breadcrumb.ResourceBreadcrumbBuilder;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.pages.AbstractPageController;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.AddressForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.UpdatePasswordForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.validation.AddressValidator;
import de.hybris.platform.acceleratorstorefrontcommons.util.AddressDataUtil;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.i18n.I18NFacade;
import de.hybris.platform.commercefacades.order.CheckoutFacade;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.enums.CountryType;
import de.hybris.platform.servicelayer.session.SessionService;
import org.apache.log4j.Logger;
import org.springframework.beans.BeanUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.Year;
import java.util.AbstractMap;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static de.hybris.platform.util.localization.Localization.getLocalizedString;


@Controller
@RequestMapping("/my-account")
public class PayPalAccountPageController extends AbstractPageController {
    private static final Logger LOG = Logger.getLogger(PayPalAccountPageController.class);

    private static final String REDIRECT_TO_PAYMENT_INFO_PAGE = REDIRECT_PREFIX + "/my-account/payment-details";
    private static final String ADD_EDIT_PAYMENT_METHOD_CMS_PAGE = "add-edit-payment-method";
    private static final String FORM_GLOBAL_ERROR = "invalid.payment.details.error";
    private static final String UPDATE_PASSWORD_CMS_PAGE = "updatePassword";
    private static final String BREADCRUMBS_ATTR = "breadcrumbs";
    private static final String REDIRECT_TO_PASSWORD_UPDATE_PAGE = REDIRECT_PREFIX + "/my-account/update-password";

    private static final String REGIONS_ATTR = "regions";
    private static final String ADDRESS_FORM = "addressForm";
    private static final String COUNTRY_ATTR = "country";

    private static final String PAYMENT_REMOVED_SUCCESSFULLY = "text.account.profile.paymentMethod.removed";
    private static final String SOMETHING_WENT_WRONG = "error.something.went.wrong";
    private static final String ERROR_DURING_DELETING_PAYMENT_METHOD = "Error during deleting payment method";
    private static final String EXPIRY_YEAR = "expiryYear";
    private static final String EXPIRY_MONTH = "expiryMonth";
    private static final String CREDIT_CARD_EXPIRY_MONTH_INVALID = "credit.card.expiry.month.invalid";
    private static final String CREDIT_CARD_EXPIRY_YEAR_INVALID = "credit.card.expiry.year.invalid";
    private static final String ERROR_ISSUE_YEAR = "YEAR";
    private static final String ERROR_ISSUE_MONTH = "MONTH";
    private static final int YEARS_QUANTITY = 20;
    private static final String ERROR_ISSUE_NUMBER = "NUMBER";
    private static final String CARD_NUMBER = "cardNumber";
    private static final String CREDIT_CARD_NUMBER_INVALID = "credit.card.number.invalid";
    private static final String ERROR_ISSUE_NAME = "NAME";
    private static final String NAME_ON_CARD = "nameOnCard";
    private static final String CREDIT_CARD_NAME_INVALID = "credit.card.name.invalid";
    private static final String UNSUCCESS_UNLINK_NOTIFICATION = "paypal.unsuccess.unlink.notification";
    private static final String SUCCESS_UNLINK_NOTIFICATION = "paypal.success.unlink.notification";
    private static final String NO_HANDLER_FOUND_FOR_PAGE_TYPE_MESSAGE = "No handler found for %s page type";

    private static final String PAYER_ACTION_REQUIRED = "PAYER_ACTION_REQUIRED";
    private static final String PAYPAL_CONNECT_SAVED_PAYMENT_METHOD_ERROR = "paypal.connect.saved.payment.method.error";
    private static final String PAYPAL_CONNECT_SAVED_PAYMENT_METHOD = "paypal.connect.saved.payment.method";
    private static final String PAYPAL_CONNECT_SAVED_PAYMENT_METHOD_VERIFICATION_ERROR =
            "paypal.connect.saved.payment.method.verification.error";
    private static final String IS_SAVED_CREDIT_CARDS_AMOUNT_LIMITED = "isSavedCreditCardsAmountLimited";
    private static final String IS_SAVED_PAYPAL_ACCOUNTS_AMOUNT_LIMITED = "isSavedPayPalAccountsAmountLimited";
    private static final String SHOW_DUPLICATE_NOTIFICATION = "SHOW_DUPLICATE_NOTIFICATION";

    @Resource(name = "userFacade")
    private UserFacade userFacade;

    @Resource(name = "payPalCustomerFacade")
    private PayPalCustomerFacade payPalCustomerFacade;

    @Resource(name = "accountBreadcrumbBuilder")
    private ResourceBreadcrumbBuilder accountBreadcrumbBuilder;

    @Resource(name ="defaultPayPalAcceleratorCheckoutFacade")
    private PayPalAcceleratorCheckoutFacade acceleratorCheckoutFacade;

    @Resource(name = "paypalConfigurationFacade")
    private PayPalConfigurationFacade payPalConfigurationFacade;

    @Resource(name = "payPalSetUpPasswordFacade")
    private DefaultPayPalSetUpPasswordFacade defaultPayPalSetUpPasswordFacade;
    @Resource
    private DefaultPayPalCreditCardFacade payPalCreditCardFacade;
    @Resource(name = "payPalPasswordValidator")
    private PayPalPasswordValidator payPalPasswordValidator;

    @Resource(name = "acceleratorCheckoutFacade")
    private CheckoutFacade checkoutFacade;
    @Resource
    private AddressValidator addressValidator;
    @Resource
    private DefaultPayPalPaymentInfoService paymentInfoService;
    @Resource
    private I18NFacade i18NFacade;
    @Resource
    private PayPalConfigurationService payPalConfigurationService;
    @Resource
    private AddressDataUtil addressDataUtil;
    @Resource
    private PaypalExpiredDateValidator paypalExpirationDateValidator;
    @Resource(name = "pageConfigurationStrategies")
    private List<PageConfigurationStrategy> pageConfigurationStrategies;
    @Resource
    private PayPalPaymentInfoService payPalPaymentInfoService;
    @Resource
    private DefaultPayPalB2BCreditCardFacade payPalB2BCreditCardFacade;
    @Resource
    private SessionService sessionService;

    @RequestMapping(value = "/add-payment-method", method = RequestMethod.GET)
    @RequireHardLogIn
    public String addPaymentMethod(final Model model) throws CMSItemNotFoundException {
        if (paymentInfoService.isSavedCreditCardsAmountLimited() && paymentInfoService.isSavedPayPalAccountsAmountLimited()) {
            return REDIRECT_TO_PAYMENT_INFO_PAGE;
        }
        populateAddPaymentMethodPage(model);
        model.addAttribute(ADDRESS_FORM, new CreditCardAddressForm());
        model.addAttribute(IS_SAVED_CREDIT_CARDS_AMOUNT_LIMITED, paymentInfoService.isSavedCreditCardsAmountLimited());
        model.addAttribute(IS_SAVED_PAYPAL_ACCOUNTS_AMOUNT_LIMITED, paymentInfoService.isSavedPayPalAccountsAmountLimited());
        return getViewForPage(model);
    }

    @RequestMapping(value = "/unlink-paypal/verify", method = RequestMethod.GET)
    @RequireHardLogIn
    @ResponseBody
    public PayPalPayerIdStatusData verifyUnlinkLoginWithPayPal() {
        final PayPalPayerIdStatusData payPalPayerIdStatusData = new PayPalPayerIdStatusData();

        payPalPayerIdStatusData.setIsPasswordForSingleAuthenticationActive(!defaultPayPalSetUpPasswordFacade.isSetUpPasswordForm());

        return payPalPayerIdStatusData;
    }

    @RequestMapping(value = "/unlink-paypal/unlink", method = RequestMethod.POST)
    @ResponseStatus(HttpStatus.OK)
    public void unlinkLoginWithPayPal() {
        payPalCustomerFacade.unlinkLoginWithPayPal();
    }

    @RequestMapping(value = "/unlink-paypal", method = RequestMethod.GET)
    @RequireHardLogIn
    public String showUnlinkLoginWithPayPalResult(final RedirectAttributes redirectAttributes, final HttpServletRequest request,
                                                  @RequestParam(value = "url", required = false) String url, @RequestParam(value = "isUnlinked") boolean unlinkStatus) {
        String redirectUrl = Optional.ofNullable(url).orElse(request.getHeader(HttpHeaders.REFERER));
        if (unlinkStatus) {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                SUCCESS_UNLINK_NOTIFICATION);
        } else {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER,
                UNSUCCESS_UNLINK_NOTIFICATION);
        }
        return REDIRECT_PREFIX + redirectUrl;
    }

    private void populateAddPaymentMethodPage(Model model) throws CMSItemNotFoundException {
        final List<Breadcrumb> breadcrumbs = accountBreadcrumbBuilder.getBreadcrumbs(null);
        breadcrumbs.add(new Breadcrumb("/my-account/payment-details",
                getMessageSource().getMessage("text.account.paymentDetails", null, getI18nService().getCurrentLocale()),
                null));
        breadcrumbs.add(new Breadcrumb("#",
                getMessageSource()
                        .getMessage("text.account.profile.paymentMethod.add", null, getI18nService().getCurrentLocale()),
                null));

        storeCmsPageInModel(model, getContentPageForLabelOrId(ADD_EDIT_PAYMENT_METHOD_CMS_PAGE));
        setUpMetaDataForContentPage(model, getContentPageForLabelOrId(ADD_EDIT_PAYMENT_METHOD_CMS_PAGE));
        model.addAttribute("breadcrumbs", breadcrumbs);
        model.addAttribute("metaRobots", "noindex,nofollow");
        model.addAttribute("paypalMyAccountConfigurationData", selectPageConfigurationStrategy(PayPalPageType.ACCOUNT).getConfiguration());
        model.addAttribute("deliveryAddresses", userFacade.getAddressBook());
        model.addAttribute("months", generateMonths());
        model.addAttribute("years", generateYears());
        model.addAttribute("supportedCountries", checkoutFacade.getCountries(CountryType.SHIPPING));
    }

    @RequestMapping(value = "/add-payment-method", method = RequestMethod.POST,
            consumes = {MediaType.APPLICATION_FORM_URLENCODED_VALUE})
    @RequireHardLogIn
    public String addCCPaymentMethod(RedirectAttributes redirectAttributes,
                                     @ModelAttribute(ADDRESS_FORM) CreditCardAddressForm paymentDetailsForm,
                                     BindingResult bindingResult, Model model) {
        try {
            paypalExpirationDateValidator.validate(paymentDetailsForm, bindingResult);
            CreditCardAddressData creditCardAddressData = new CreditCardAddressData();
            BeanUtils.copyProperties(paymentDetailsForm, creditCardAddressData);
            creditCardAddressData.setCardNumber(paymentDetailsForm.getCardNumber().replaceAll("\\s+",""));

            boolean isNewAddressShouldBeCreated = paymentDetailsForm.getSelectedAddressCode() == null;
            if (isNewAddressShouldBeCreated) {
                AddressForm addressForm = validateAddress(paymentDetailsForm, bindingResult);
                if (!bindingResult.hasErrors()) {
                    AddressData newBillingAddress = addressDataUtil.convertToAddressData(addressForm);
                    creditCardAddressData.setBillingAddress(newBillingAddress);
                    newBillingAddress.setVisibleInAddressBook(false);
                    newBillingAddress.setShippingAddress(true);
                    userFacade.addAddress(newBillingAddress);

                    creditCardAddressData.setSelectedAddressCode(newBillingAddress.getId());
                }
            } else {
                AddressData addressForCode = userFacade.getAddressForCode(creditCardAddressData.getSelectedAddressCode());
                creditCardAddressData.setBillingAddress(addressForCode);
                sessionService.setAttribute(SHOW_DUPLICATE_NOTIFICATION, Boolean.TRUE);
            }

            if (bindingResult.hasErrors()) {
                populatePageAfterFailedValidation(paymentDetailsForm, model);
                return getViewForPage(model);
            }

            if (paymentDetailsForm.isAddBillingAddressToAddressBook()) {
                AddressData newShippingAddress = getAddressDataFromPaymentDetailsForm(paymentDetailsForm);
                if (!payPalCustomerFacade.addShippingAddressToCurrentUser(newShippingAddress)) {
                    GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER,
                            getLocalizedString(Paypalb2baddonWebConstants.CREDIT_CARD_ADD_BILLING_ADDRESS_TO_ADDRESS_BOOK_DUPLICATE_NOTIFICATION));
                    sessionService.setAttribute(SHOW_DUPLICATE_NOTIFICATION, Boolean.TRUE);
                }
            }

            SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
            setupTokenRequestData.setCreditCardData(creditCardAddressData);
            setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);

            CCSetupTokenData ccSetupTokenData = payPalCreditCardFacade.requestSetupToken(setupTokenRequestData);

            if (ccSetupTokenData.getStatus().equals(PAYER_ACTION_REQUIRED)) {
                return REDIRECT_PREFIX + ccSetupTokenData.getApproveLink();
            } else {
                PaymentTokenData paymentTokenData = preparePaypalTokenDataForRequest(ccSetupTokenData);
                processRequestForPaymentToken(redirectAttributes, paymentTokenData);
            }
        } catch (HttpClientErrorException ex) {
            LOG.error(ex.getMessage(), ex);
            String exceptionResponseBody = ex.getResponseBodyAsString();
            LOG.error(exceptionResponseBody);

            populateValidationErrors(bindingResult, exceptionResponseBody);
            populatePageAfterFailedValidation(paymentDetailsForm, model);

            return getViewForPage(model);
        } catch (Exception e) {
            LOG.error("Error during adding payment method for customer", e);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER, PAYPAL_CONNECT_SAVED_PAYMENT_METHOD_ERROR);
        }
        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    private AddressData getAddressDataFromPaymentDetailsForm(CreditCardAddressForm paymentDetailsForm) {
        AddressForm addressForm = new AddressForm();
        BeanUtils.copyProperties(paymentDetailsForm, addressForm);
        return addressDataUtil.convertToAddressData(addressForm);
    }

    private void processRequestForPaymentToken(RedirectAttributes redirectAttributes, PaymentTokenData paymentTokenData) throws Exception {
        payPalCreditCardFacade.requestPaymentToken(paymentTokenData);

        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                PAYPAL_CONNECT_SAVED_PAYMENT_METHOD);
    }

    @RequestMapping(value = "/payment-details/approved", method = RequestMethod.GET)
    @RequireHardLogIn
    public String processWithPaymentTokenForApprovedSetupToken(RedirectAttributes redirectAttributes, @RequestParam String guid) {
        try {
            PayPalCreditCardPaymentInfoModel paymentInfoByCode = payPalPaymentInfoService.getPaymentInfoByPK(guid)
                    .orElseThrow(Exception::new);

            PaymentTokenData paymentTokenData = preparePaypalTokenDataForRequest(paymentInfoByCode);
            payPalCreditCardFacade.requestPaymentToken(paymentTokenData);

            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                    PAYPAL_CONNECT_SAVED_PAYMENT_METHOD);

            if ((Boolean) sessionService.getAttribute(SHOW_DUPLICATE_NOTIFICATION)) {
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER,
                        getLocalizedString(Paypalb2baddonWebConstants.CREDIT_CARD_ADD_BILLING_ADDRESS_TO_ADDRESS_BOOK_DUPLICATE_NOTIFICATION));
            }

        } catch (Exception e) {
            LOG.error("Error during requesting payment token for customer", e);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    PAYPAL_CONNECT_SAVED_PAYMENT_METHOD_VERIFICATION_ERROR);
        }
        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    @RequestMapping(value = "/payment-details/cancelled", method = RequestMethod.GET)
    @RequireHardLogIn
    public String processWithCancelledSetupToken(RedirectAttributes redirectAttributes, @RequestParam String guid) {
        payPalPaymentInfoService.removePaymentInfoByPK(guid);

        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                PAYPAL_CONNECT_SAVED_PAYMENT_METHOD_VERIFICATION_ERROR);

        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    private void populateValidationErrors(BindingResult bindingResult, String exceptionResponseBody) {
        PaypalValidationError paypalValidationErrors = new Gson().fromJson(exceptionResponseBody,
                PaypalValidationError.class);
        paypalValidationErrors.getDetails().forEach(errorDetail -> {
            if (errorDetail.getIssue().contains(ERROR_ISSUE_YEAR)) {
                bindingResult.rejectValue(EXPIRY_YEAR, CREDIT_CARD_EXPIRY_YEAR_INVALID);
            } else if (errorDetail.getIssue().contains(ERROR_ISSUE_MONTH)) {
                bindingResult.rejectValue(EXPIRY_MONTH, CREDIT_CARD_EXPIRY_MONTH_INVALID);
            } else if (errorDetail.getIssue().contains(ERROR_ISSUE_NUMBER)) {
                bindingResult.rejectValue(CARD_NUMBER, CREDIT_CARD_NUMBER_INVALID);
            } else if (errorDetail.getIssue().contains(ERROR_ISSUE_NAME)) {
                bindingResult.rejectValue(NAME_ON_CARD, CREDIT_CARD_NAME_INVALID);
            }
        });
    }

    private PaymentTokenData preparePaypalTokenDataForRequest(CCSetupTokenData ccSetupTokenData) {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setSetupToken(ccSetupTokenData.getId());
        paymentTokenData.setAddressId(ccSetupTokenData.getSelectedBillingAddress());
        paymentTokenData.setPaymentInfoPK(ccSetupTokenData.getPaymentInfoPK());
        paymentTokenData.setCustomerId(ccSetupTokenData.getCustomerId());

        return paymentTokenData;
    }

    private PaymentTokenData preparePaypalTokenDataForRequest(PayPalCreditCardPaymentInfoModel creditCardPaymentInfo) {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setSetupToken(creditCardPaymentInfo.getSubscriptionId());
        paymentTokenData.setAddressId(creditCardPaymentInfo.getBillingAddress().getPk().getLongValueAsString());
        paymentTokenData.setPaymentInfoPK(creditCardPaymentInfo.getPk().getLongValueAsString());

        return paymentTokenData;
    }

    private AddressForm validateAddress(CreditCardAddressForm paymentDetailsForm, BindingResult bindingResult) {
        AddressForm addressForm = new AddressForm();
        BeanUtils.copyProperties(paymentDetailsForm, addressForm);
        addressValidator.validate(addressForm, bindingResult);
        return addressForm;
    }

    private void populatePageAfterFailedValidation(CreditCardAddressForm paymentDetailsForm, Model model) {
        GlobalMessages.addErrorMessage(model, FORM_GLOBAL_ERROR);
        try {
            populateAddPaymentMethodPage(model);
        } catch (CMSItemNotFoundException e) {
            throw new PayPalCMSItemNotFoundException(e);
        }
        if (paymentDetailsForm.getCountryIso() != null) {
            model.addAttribute(COUNTRY_ATTR, paymentDetailsForm.getCountryIso());
            model.addAttribute(REGIONS_ATTR, i18NFacade.getRegionsForCountryIso(paymentDetailsForm.getCountryIso()));
        }
    }

    @RequestMapping(value = "/delete-payment-method", method = RequestMethod.POST)
    @RequireHardLogIn
    public String deletePaymentMethod(@RequestParam(value = "paymentInfoId") final String paymentMethodId,
                                      final RedirectAttributes redirectAttributes)
    {
        try {
            payPalCreditCardFacade.deleteCreditCardFromPayPal(paymentMethodId);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                    PAYMENT_REMOVED_SUCCESSFULLY);
        } catch (PayPalCreditCardRemovalException e) {
            LOG.error(ERROR_DURING_DELETING_PAYMENT_METHOD, e);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    SOMETHING_WENT_WRONG);
        }
        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    @RequestMapping(value = "/set-password", method = RequestMethod.POST)
    @RequireHardLogIn
    public String setPassword(final UpdatePasswordForm updatePasswordForm, final BindingResult bindingResult,
        final Model model,
        final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        payPalPasswordValidator.validate(updatePasswordForm, bindingResult);
        if (!bindingResult.hasErrors()) {
            if (updatePasswordForm.getNewPassword().equals(updatePasswordForm.getCheckNewPassword())) {
                defaultPayPalSetUpPasswordFacade.setUpPassword(updatePasswordForm.getNewPassword());
            } else {
                bindingResult.rejectValue("checkNewPassword", "validation.checkPwd.equals", new Object[]{},
                    "validation.checkPwd.equals");
            }
        }

        if (bindingResult.hasErrors()) {
            GlobalMessages.addErrorMessage(model, FORM_GLOBAL_ERROR);
            final ContentPageModel updatePasswordPage = getContentPageForLabelOrId(UPDATE_PASSWORD_CMS_PAGE);
            storeCmsPageInModel(model, updatePasswordPage);
            setUpMetaDataForContentPage(model, updatePasswordPage);

            model.addAttribute(BREADCRUMBS_ATTR,
                accountBreadcrumbBuilder.getBreadcrumbs("text.account.profile.updatePasswordForm"));
            return getViewForPage(model);
        } else {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                "text.account.confirmation.password.set", null);
            return REDIRECT_TO_PASSWORD_UPDATE_PAGE;
        }
    }

    private List<AbstractMap.SimpleEntry<Integer, Integer>> generateYears() {
        Year currentYear = Year.now();
        return IntStream.rangeClosed(currentYear.getValue(), currentYear.plusYears(YEARS_QUANTITY).getValue())
                .boxed()
                .map(yearVal -> new AbstractMap.SimpleEntry<>(yearVal, yearVal))
                .collect(Collectors.toList());
    }


    private List<AbstractMap.SimpleEntry<String, String>> generateMonths() {
        return  IntStream.rangeClosed(1, 12)
                .mapToObj(num -> String.format("%02d", num))
                .map(formattedVal -> new AbstractMap.SimpleEntry<>(formattedVal, formattedVal))
                .collect(Collectors.toList());
    }


    private PageConfigurationStrategy selectPageConfigurationStrategy(PayPalPageType pageType) {
        return pageConfigurationStrategies.stream()
                .filter(strategy -> strategy.test(pageType))
                .findAny()
                .orElseThrow(() -> new IllegalArgumentException(String.format(NO_HANDLER_FOUND_FOR_PAGE_TYPE_MESSAGE, pageType)));
    }
}
