package com.paypal.hybris.addon.validators;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorstorefrontcommons.forms.UpdateEmailForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.UpdatePasswordForm;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.validation.Errors;

@UnitTest
public class PayPalPasswordValidatorTest {

    private static final String NEW_PASSWORD = "newPassword";
    private static final String UPDATE_PWD_INVALID = "updatePwd.pwd.invalid";
    private static final String CHECK_NEW_PASSWORD = "checkNewPassword";

    @Mock
    private Errors errors;

    @InjectMocks
    private PayPalPasswordValidator unit;

    private UpdatePasswordForm updatePasswordForm;

    @Before
    public void setup() {
        MockitoAnnotations.initMocks(this);

        updatePasswordForm = new UpdatePasswordForm();
    }

    @Test
    public void shouldSupportClass() {
        assertTrue(unit.supports(UpdatePasswordForm.class));
    }

    @Test
    public void shouldNotSupportClass() {
        assertFalse(unit.supports(UpdateEmailForm.class));
    }

    @Test
    public void shouldRejectEmptyNewPassword() {
        updatePasswordForm.setNewPassword("");
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldRejectShortNewPassword() {
        updatePasswordForm.setNewPassword("12345");
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldRejectLongNewPassword() {
        updatePasswordForm.setNewPassword(StringUtils.repeat('a', 256));
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldAcceptValidNewPassword() {
        updatePasswordForm.setNewPassword("validPass123");
        unit.validate(updatePasswordForm, errors);
        verify(errors, never()).rejectValue(eq(NEW_PASSWORD), any());
    }

    @Test
    public void shouldRejectEmptyCheckNewPassword() {
        updatePasswordForm.setCheckNewPassword("");
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(CHECK_NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldRejectShortCheckNewPassword() {
        updatePasswordForm.setCheckNewPassword("12345");
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(CHECK_NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldRejectLongCheckNewPassword() {
        updatePasswordForm.setCheckNewPassword(StringUtils.repeat('a', 256));
        unit.validate(updatePasswordForm, errors);
        verify(errors).rejectValue(CHECK_NEW_PASSWORD, UPDATE_PWD_INVALID);
    }

    @Test
    public void shouldAcceptValidCheckNewPassword() {
        updatePasswordForm.setCheckNewPassword("validPass123");
        unit.validate(updatePasswordForm, errors);
        verify(errors, never()).rejectValue(eq(CHECK_NEW_PASSWORD), any());
    }
}